/*******************************************************************************
 *    uart2.c
 *   PIC32MX/PIC32MZ uart driver
 *   2025.3.3  Suwa-Koubou
 */

#include <string.h>
#include "HardwareProfile.h"
#include "uart2.h"

#define BAUDRATEREG2        (GetPeripheralClock2()/BRG_DIV2/BAUDRATE2-1L)

#define RXBUFSIZE   1024      
#define TXBUFSIZE   4096      
static char RxBuf[RXBUFSIZE];
static char TxBuf[TXBUFSIZE];
static volatile int TxWrtPtr;
static volatile int TxRdPtr;
static volatile int TxSize;
static volatile int RxWrtPtr;
static volatile int RxRdPtr;
static volatile int RxSize;


// Override _mon_putc to allow printf to send characters over UART2
void _mon_putc (char c)
{
    if(c=='\n') UART2PutChar('\r');
    UART2PutChar(c);
}


/*******************************************************************************
     UART2Init()
*******************************************************************************/

void UART2Init()
{
    U2BRG = BAUDRATEREG2;
    U2MODE = 0;
    U2MODEbits.BRGH = BRGH2;
    U2STA = 0;
    U2MODEbits.UARTEN = 1;

    U2STAbits.UTXISEL = 0;
    U2STAbits.URXISEL = 0;
    U2STAbits.UTXEN = 1;
    U2STAbits.URXEN = 1;
    IFS4bits.U2RXIF = 0;
    IFS4bits.U2TXIF = 0;
    IFS4bits.U2EIF = 0;

	RxRdPtr = RxWrtPtr = RxSize = 0;
	TxRdPtr = TxWrtPtr = TxSize = 0;

	IPC36bits.U2RXIP = 3;   // UART 2 interrupt priority is 3
	IPC36bits.U2RXIS = 1;   // UART 2 interrupt priority is 3
	IPC36bits.U2TXIP = 3;   // UART 2 interrupt priority is 3
	IPC36bits.U2TXIS = 2;   // UART 2 interrupt priority is 3

	IEC4bits.U2RXIE = 1; // 1; // Enable RX interrupt
//	IEC4bits.U2TXIE = 1; // do not enable this timing
	IEC4bits.U2EIE =1;   // Enable ERR interrupt
}


/*******************************************************************************
    UART2GetTxSize()
*******************************************************************************/

int UART2GetTxSize(void)
{
    return TxSize;
}


/*******************************************************************************
    UART2GetTrmtStat()
*******************************************************************************/

int UART2GetTrmtStat(void)
{
    return (U2STAbits.TRMT ? 0: 0x10000);
}


/*******************************************************************************
    UART2TxBufClr()
*******************************************************************************/

void UART2TxBufClr(void)
{
	IEC4bits.U2TXIE=0;
	TxRdPtr = TxWrtPtr = TxSize = 0;	
	IEC4bits.U2TXIE=1;
}

/*******************************************************************************
    UART2PutChar()
*******************************************************************************/

int UART2PutChar(char ch)
{
    if(TxSize >= TXBUFSIZE) return -1; // buffer full

 	TxBuf[TxWrtPtr] = ch;
	TxWrtPtr++;
	TxWrtPtr %= TXBUFSIZE;
	IEC4bits.U2TXIE = 0; // lock interrupt
    TxSize++;
	IEC4bits.U2TXIE = 1; // lock interrupt
    return 0;
}

void UART2PutChar_wait(char ch)
{
    while(UART2PutChar(ch)==-1);
}


/*******************************************************************************
    UART2PrintString()
*******************************************************************************/

void UART2PrintString( char *str )
{
    while(*str){
        UART2PutChar_wait(*str);
        str++;
    }
}

/*******************************************************************************
    UART2PutData()
*******************************************************************************/

int UART2PutData(char *data, int len)
{
    int size;

    if((TXBUFSIZE - TxSize) < len) return -1;

    IEC4bits.U2TXIE = 0; // lock interrupt
    size = TXBUFSIZE - TxWrtPtr;
    if(size >= len){
        memcpy(&TxBuf[TxWrtPtr], data, len);
        TxWrtPtr += len;
    }else{
        memcpy(&TxBuf[TxWrtPtr], data, size);
        memcpy(&TxBuf[0], &data[size], len-size);
        TxWrtPtr = len-size;
    }
    TxWrtPtr %= TXBUFSIZE;
    TxSize += len;
    IEC4bits.U2TXIE = 1; // unlock interrupt

    return 0;
}

void UART2PutData_wait(char *data, int len)
{
    while(UART2PutData(data, len) == -1);
}


/*******************************************************************************
    UART2IsPressed()
    UART2GetRxSize()
*******************************************************************************/

int UART2IsPressed()
{
	return RxSize;
}


int UART2GetRxSize(void)
{
	return RxSize;
}


/*******************************************************************************
    UART2RxBufClr()
*******************************************************************************/

void UART2RxBufClr(void)
{
	IEC4bits.U2RXIE=0;
	RxRdPtr = RxWrtPtr = RxSize = 0;	
	IEC4bits.U2RXIE=1;
}


/*******************************************************************************
    UART2GetChar()
*******************************************************************************/

int UART2GetChar()
{
	char c;

    while(RxSize == 0){
       ;  // wait for received data
	}
    c = RxBuf[RxRdPtr];
	RxRdPtr++;
	RxRdPtr %= RXBUFSIZE;

   	IEC4bits.U2RXIE=0;
    RxSize--;
	IEC4bits.U2RXIE=1;

	return c;
}

/*******************************************************************************
    UART2GetString() 
*******************************************************************************/

char *UART2GetString(char *buf, int size)
{
	int c, len;

	len=0;

	while(len<(size-1)){
		c = UART2GetChar();
        if(c == 0x03){ // ctrl+'C'
            buf[0] =0;
            return NULL;
        }
        UART2PutChar(c);
		if(c == '\r'){
            UART2PutChar('\n');
			buf[len] = 0;
			return buf;

		}else if(c == 0x08){ // back space
            UART2PutChar(0x08); 
            len--;
        }else{      
		    buf[len++]=c;
	    }
    }   
	buf[len]=0;

	return buf;
}

/*******************************************************************************
    UART2GetData() 
*******************************************************************************/

void UART2GetData(char *data, int len)
{
	while(len--){
		*data++ = UART2GetChar();
	}
}


/*******************************************************************************
    PIC32MZ uart interrupt function
*******************************************************************************/

void __attribute__((interrupt(ipl3srs), vector(_UART2_RX_VECTOR) )) _U2RXInterrupt(void)
{
	// error interrupt
	if(IFS4bits.U2EIF){
		IFS4bits.U2EIF=0;
		if(U2STAbits.OERR){
			U2STAbits.OERR=0;
		}
		if(U2STAbits.FERR){
			U2STAbits.FERR=0;
		}
        if(IFS4bits.U2RXIF){
            IFS4bits.U2RXIF=0;
        }
	}

	// RXinterrupt ?
	if(IEC4bits.U2RXIE && IFS4bits.U2RXIF && U2STAbits.URXDA){
		char c;

		c = U2RXREG;
        // if buffer empty, store data to rx buffer.
		if(RxSize < RXBUFSIZE){
			RxBuf[RxWrtPtr++]=c;
			RxWrtPtr %= RXBUFSIZE;
            RxSize++;
		}
		if(!U2STAbits.URXDA) IFS4bits.U2RXIF = 0;
	}

}

void __attribute__((interrupt(ipl3srs), vector(_UART2_TX_VECTOR) )) _U2TXInterrupt(void)
{
    // TXinterrupt ?
	if(IEC4bits.U2TXIE && IFS4bits.U2TXIF){

		while(U2STAbits.UTXBF);
		IFS4bits.U2TXIF = 0;

        if(TxSize == 0){
			IEC4bits.U2TXIE = 0; // disable TX interrupt
			return; // no data in TxBuf
		}

		U2TXREG = TxBuf[TxRdPtr++];
		TxRdPtr %= TXBUFSIZE;
        TxSize--;
	}
}



/*******************************************************************************
    UART2PutDec()
*******************************************************************************/

void UART2PutDec(int val)
{
    int divide;
    int digit;
    int zflg;

    if(val == 0x80000000){
        UART2PrintString("-2147483648");
        return;
    }

    if(val < 0){
        UART2PutChar('-');
        val = -val;
    }
    divide = 1000000000;
    zflg=0;

    for(;;){
        if(divide ==1){
        	UART2PutChar(val + '0');
            break;
        }
        digit = val / divide;
        if(digit==0){
            if(zflg){
            	UART2PutChar('0');
            }
        }else{
        	UART2PutChar(digit +'0');
            zflg=1;
        }
        val %= divide;
        divide /= 10;
    }

}

/*******************************************************************************
    UART2PutHex()
*******************************************************************************/

static const unsigned char CharacterArray[]={'0','1','2','3','4','5','6','7','8','9','A','B','C','D','E','F'};

void UART2PutHex( int toPrint )
{
    int printVar;

    printVar = toPrint;
    toPrint = (toPrint>>4) & 0x0F;
    UART2PutChar( CharacterArray[toPrint] );

    toPrint = printVar & 0x0F;
    UART2PutChar( CharacterArray[toPrint] );

    return;
}

/*******************************************************************************
    UART1PutHexWord()
    UART1PutHexDWord()
*******************************************************************************/

void UART2PutHexWord( unsigned int toPrint )
{
    unsigned int printVar;

    printVar = (toPrint>>12) & 0x0F;
    UART2PutChar( CharacterArray[printVar] );

    printVar = (toPrint>>8) & 0x0F;
    UART2PutChar( CharacterArray[printVar] );

    printVar = (toPrint>>4) & 0x0F;
    UART2PutChar( CharacterArray[printVar] );

    printVar = toPrint & 0x0F;
    UART2PutChar( CharacterArray[printVar] );

    return;
}

void UART2PutHexDWord( unsigned long toPrint )
{
    unsigned long printVar;

    printVar = (toPrint>>28) & 0x0F;
    UART2PutChar( CharacterArray[printVar] );

    printVar = (toPrint>>24) & 0x0F;
    UART2PutChar( CharacterArray[printVar] );

    printVar = (toPrint>>20) & 0x0F;
    UART2PutChar( CharacterArray[printVar] );

    printVar = (toPrint>>16) & 0x0F;
    UART2PutChar( CharacterArray[printVar] );

    printVar = (toPrint>>12) & 0x0F;
    UART2PutChar( CharacterArray[printVar] );

    printVar = (toPrint>>8) & 0x0F;
    UART2PutChar( CharacterArray[printVar] );

    printVar = (toPrint>>4) & 0x0F;
    UART2PutChar( CharacterArray[printVar] );

    printVar = toPrint & 0x0F;
    UART2PutChar( CharacterArray[printVar] );

    return;
}


/*******************************************************************************
    UART2Char2Hex()
*******************************************************************************/

char UART2Char2Hex(char ch){
  // Wrong char
  if(ch > 102)
    return 0;  
  
  // From a to f     
  if(ch > 96)
    return (ch-87);
    
  // Wrong char    
  if(ch > 70)
    return 0;
  
  // From A to F    
  if(ch > 64)
    return (ch-55);

  // Wrong char  
  if(ch > 57)
    return 0;
  
  // From 0 - 9
  if(ch > 47)
    return(ch-48);
  else
  // Wrong char
    return 0;
}

/*******************************************************************************
    UART2Hex2Char()
*******************************************************************************/

char UART2Hex2Char(char hex)
{
    char h;
    h = hex&0x0f;
    // From 0xa to 0xf
    if(h>9)
        return (h+55);
    else
        return (h+48);
}

/*** end of uart2.c ***********************************************************/