/*********************************************************************
	timer.c
	
	Application timer module
   (c) 2013 Suwa-Koubou
**********************************************************************/

#include   "HardwareProfile.h"
#include   "timer.h"

// fatfs rtc
extern void rtc_clock(void);

/*-----------------------------------------------------------------------
	definition
-----------------------------------------------------------------------*/

#define	_DI()	IEC0bits.T1IE = 0		// disable T1 interrupt
#define	_EI()	IEC0bits.T1IE = 1		// enable T1 interrupt

/*-----------------------------------------------------------------------
	global variable
-----------------------------------------------------------------------*/

// for application timer
static	volatile unsigned int tm[MAX_TIMER]; 
static	volatile unsigned char  tuse[MAX_TIMER];

static unsigned int  systick = 0;


/*-----------------------------------------------------------------------
	initilaize timer module
-----------------------------------------------------------------------*/

void timer_init(void)
{
	int	i;

	// clear timer table
	for(i = 0; i < MAX_TIMER; i++) {
		tm[i] = tuse[i] = 0;
	}
    
    tuse[0] = 1;  // ^C}[ԍ OԂ͕sgpƂ. 2017.9.4
    
	//--- setting interrupt ---------------------------------------
    IEC0SET = _IEC0_T1IE_MASK; // enable T1 interrupt
    IFS0CLR = _IFS0_T1IF_MASK; // clear interrupt flag
    IPC1bits.T1IP = 1;

    //--- set interval@1ms timer
    T1CON = 0x0010;	// timer(T1), set prescale 1:8
    TMR1 = 0;
    PR1 = GetPeripheralClock3()/8/1000L -1;    // period 1msec 
    T1CONSET = _T1CON_TON_MASK;    // start T1
}


/*----------------------------------------------------------------------
	Timer T1 1msec interval timer.
----------------------------------------------------------------------*/

void __attribute__ (( interrupt(IPL1SRS), vector(_TIMER_1_VECTOR) )) _T1Interrupt(void)
{
	int	i;

    //LATCbits.LATC1 ^= 1;

	//IFS0bits.T1IF = 0;		// clear interrupt flag
    IFS0CLR = _IFS0_T1IF_MASK; // clear interrupt flag

    // system timer tick
    systick++;

    // fatfs
    rtc_clock();

	// timer count down
	for(i = 1; i < MAX_TIMER; i++){
		if(tm[i]) tm[i]--;
	}

}


unsigned int gettick(void)
{
    return systick;
}


/*----------------------------------------------------------------------
	timer_entry()
----------------------------------------------------------------------*/

int	timer_entry(unsigned int tm_val)
{
	int n;

	for(n = 1; n < MAX_TIMER; n++){
		if(tuse[n] == 0){ 
			tuse[n] = 1;
			_DI();
			tm[n] = tm_val;
			_EI();
			return n;
		}
	}

    //putstr("not enough timer!\r\n");
	return -1;
}


/*----------------------------------------------------------------------
	timer_update()
----------------------------------------------------------------------*/

void timer_update(int n, unsigned int tm_val)
{
	if(n >= 1 && n < MAX_TIMER){
		_DI();
		tm[n] = tm_val;
		_EI();
	}
}


/*----------------------------------------------------------------------
	timer_delete()
----------------------------------------------------------------------*/

void timer_delete(int n)
{
	if(n >= 1 && n < MAX_TIMER){
		tuse[n] = 0;
	}
}


/*----------------------------------------------------------------------
	timer_check()
----------------------------------------------------------------------*/

int timer_check(int n)
{
	unsigned int t;

	if(n >= 1 && n < MAX_TIMER){
		_DI();
		t = tm[n];
		_EI();
		return t;
	}
	return 0;
}


/*----------------------------------------------------------------------
	timer_Up()
----------------------------------------------------------------------*/

int timer_Up(int n)
{
	unsigned int t;

	if(n >= 1 && n < MAX_TIMER){
		_DI();
		t = tm[n];
		_EI();
		return t == 0 ? 1: 0;
	}
	return 0;
}


/*** end of timer.c *************************************************/

