/***********************************************************************

	video_vga.c

	2022/07/20  S.Suwa http://www.suwa-koubou.jp

	640x480 31.5KHz 59.9Hz VESA VGA specification

***********************************************************************/

#include "GenericTypeDefs.h"
#include "HardwareProfile.h"
#include "mz1200.h"

/*----------------------------------------------------------------------
	constant definition
----------------------------------------------------------------------*/

// video signal parameter
//
#define V_RES		400     // Vertical Resolution: 25line*8dot double 
#define H_RES		320     // Horizontal Resolution: 40chr*8dot 

#define V_LINES             525     // horizontal lines/frame 
#define V_SYNC_LINES        2       // vertical sync
#define V_BLANK_LINES       (V_LINES - V_RES - V_SYNC_LINES)
#define V_FP_LINES           V_BLANK_LINES / 2          // front porch blanking
#define V_BP_LINES           V_BLANK_LINES - V_FP_LINES // back porch blanking

// video signal output stae
// horizontal
#define ST_H_FP        0       // front porch
#define ST_H_SYNC      1       // sync
#define ST_H_BP        2       // back porch
#define	ST_H_SIGNAL    3       // video signal

// vertical
#define ST_V_FP        0       // front porch
#define ST_V_SYNC      1       // sync 
#define ST_V_BP        2       // back porch
#define ST_V_SIGNAL    3       // video signal

// timing parameter
#define H_LINE_CPS     1271   // 40MHz/31.469KHz=1271
                              // 40MHz/1271=31.471KHz
                              // (1/40MHz)*1271=31.775usec
#define H_SYNC_CPS     160    // (1/40MHz)*160=4usec
#define H_BP_CPS        80    // (1/40MHz)*80=2usec

#define H_PIXEL_CLOCK   13333333L  // 13.3MHz
                              // 320*(1/13.33MHz)=24usec < (31.775-4-2)=25.775

// Signal Out Pins 
#define H_SYNC_OUT	LATDbits.LATD0		//@h_sync out
#define V_SYNC_OUT  LATDbits.LATD10     // v_sync out

#define VIDEO_OUT	LATDbits.LATD3		// video out SDO3


/*----------------------------------------------------------------------
	variable
----------------------------------------------------------------------*/

// Video memory address is from 0xD000 to 0xD7ff.
// This 2Kbyte memorys is used as ring buffer in MZ-80A. 
// Video screen hardware roll up, roll down, controled by monitor program.
static unsigned short VPtr;     // 0xD000-0xD7FF
static unsigned short VrvsPtr;  // 0-999
static unsigned short Lptr;     // save buffer of VPtr
static unsigned short LrvsPtr;  // save buffer of VrvsPtr
volatile unsigned short VMem;   // current Video Memory begin address
static unsigned char *cgrom;    // CG-ROM line pointer

static volatile unsigned int  VCount, VState, DotCount, Line;
static unsigned int nextVS[4] = {ST_V_SYNC, ST_V_BP, ST_V_SIGNAL, ST_V_FP,};   // next vertical state
static unsigned int nextVC[4] = {V_SYNC_LINES, V_BP_LINES, V_RES, V_FP_LINES}; // horizontal lines in next vertical state


/*======================================================================
	void video_init(void)
======================================================================*/

void video_init(void)
{
    // sync signal output port
    TRISDbits.TRISD0 = 0;   // VGA h_sync (OC1/RD0)
    LATDbits.LATD0 = 1;	    // output level High

    TRISDbits.TRISD10 = 0;  // VGA v_sync (RD10)
    LATDbits.LATD10 = 1;    // output level High

    // video signal output port
    TRISDbits.TRISD3 = 0; // video out(SDO3)
    LATDbits.LATD3 = 0;	  // output level low

    // set Timer2
    T2CON = 0x0010; // prescale 1:2 40MHz
    TMR2 = 0;
    PR2 = H_LINE_CPS-1;
    IEC0SET = _IEC0_T2IE_MASK; // enable interrupt
    IFS0CLR = _IFS0_T2IF_MASK; // flag clear
    IPC2bits.T2IP = 6;         //@priority 6

    // set OC1(RD0) for Horizontal sync signal
    OC1CON = 0;
    OC1R = H_SYNC_CPS;
    OC1RS = PR2;
    OC1CONbits.OCM = 0b101; // continues pulse 

    // set OC2 interrupt
    IEC0SET = _IEC0_OC2IE_MASK; // enable interrupt
    IFS0CLR = _IFS0_OC2IF_MASK; // flag clear
    IPC2bits.OC2IP = 5;         //@priority 5

    // set SPI3
    SPI3CON=0x00010834;            // SPI3 disable,32bit,master, not use SDI,
    SPI3BRG=2;	                   // 80MHz/(2*(2+1)) = 13.3MHz
    IEC0CLR = _IEC0_SPI3TXIE_MASK; // disable TX interrupt
    IFS0CLR = _IFS0_SPI3TXIF_MASK; // flag clear
    IPC6bits.SPI3IP = 5;           //@priority 5
    SPI3STATbits.SPIROV=0;         // clear ovf bit 
    SPI3CONbits.ON = 1;	           // enable SPI3

    // set video variable
    VState = ST_V_FP;			   // initialize state
    VCount = V_FP_LINES;		   // vertical counter preset
    VPtr = VMem = 0xD000;		   // set video memory begin address
    VrvsPtr = 0;
    Line = 0;                      // 0 - 399(=8x25x2)

    // go VGA
    OC1CONbits.ON = 1;
    T2CONbits.TON = 1;

}


/*----------------------------------------------------------------------
	Timer2 interrupt
----------------------------------------------------------------------*/

void __attribute__ (( interrupt(ipl6), vector(_TIMER_2_VECTOR) )) _T2Interrupt(void)
{
    // Xe[gɂ蕪
    switch(VState){
    case ST_V_BP:          // BackPorch
    case ST_V_FP:          // FrontPorch					
        V_SYNC_OUT = 1;
        break;

    case ST_V_SYNC:	       // 
        V_SYNC_OUT = 0;
        break;

    case ST_V_SIGNAL:      // f\
        if(v_gate){
            OC2R = H_SYNC_CPS; // + H_BP_CPS; // fJn܂ł̒x(SPI3̑MJnxlj
            OC2CON = 0x8001;		// ^C}2IVbg@L->H
        }
        if((Line & 0xF) == 0){
            Lptr = VPtr; // saved begin address of line
            LrvsPtr = VrvsPtr;
        }else{      
            VPtr = Lptr; // restore begin address of line
            VrvsPtr = LrvsPtr;
        }
        cgrom=cgROM[((Line++)>>1)&0x7]; // set read line
        break;
    }

    //// JE^XV
    if(--VCount == 0){		        // horizontal lines count down, state end?
        VCount = nextVC[VState];    // ̃Xe[ghorizontal lineVCountɐݒ
        VState = nextVS[VState];    // ̃Xe[g֍XV

        // Xe[gH
        if(VState == ST_V_SYNC){
            VPtr = VMem;		    // t[̍ŏɍăZbg
            VrvsPtr = 0;
            Line = 0;
        }

        if(VState==ST_V_SIGNAL){
            v_blank = 0x80;         // A no
        }else{
            v_blank = 0x00;         // A yes
        }      
    }

    IFS0CLR = _IFS0_T2IF_MASK;		// T2荞݃tONA
}


/*----------------------------------------------------------------------
	OC2 interrupt
----------------------------------------------------------------------*/

void __attribute__ (( interrupt(ipl5), vector(_OUTPUT_COMPARE_2_VECTOR) )) _OC2Interrupt(void)
{
    DotCount = 0;
    IEC0SET = _IEC0_SPI3TXIE_MASK;  // SPIM荞݂

    IFS0CLR = _IFS0_OC2IF_MASK;		// OC2荞݃tONA
}


/*----------------------------------------------------------------------
	SPI3 TX interrupt
----------------------------------------------------------------------*/

void __attribute__ (( interrupt(ipl5), vector(_SPI_3_VECTOR) )) _SPI3TXInterrupt(void)
{
    unsigned long d1,d2,d3,d4; //,data;

    //// rfIMo
    // o͎s(10)
    if(DotCount < 10){		// 40/(4/)

        d1=(cgrom[RAM[VPtr]]^Vrvs[VrvsPtr])<<24;
        d2=(cgrom[RAM[VPtr+1]]^Vrvs[VrvsPtr+1])<<16;
        d3=(cgrom[RAM[VPtr+2]]^Vrvs[VrvsPtr+2])<<8;
        d4=(cgrom[RAM[VPtr+3]]^Vrvs[VrvsPtr+3]);
        SPI3BUF = d1|d2|d3|d4;
        VPtr+=4; VrvsPtr +=4;
        VPtr &= 0xD7FF; // memory address rolling
        DotCount++;					// hbgJE^XV

        d1=(cgrom[RAM[VPtr]]^Vrvs[VrvsPtr])<<24;
        d2=(cgrom[RAM[VPtr+1]]^Vrvs[VrvsPtr+1])<<16;
        d3=(cgrom[RAM[VPtr+2]]^Vrvs[VrvsPtr+2])<<8;
        d4=(cgrom[RAM[VPtr+3]]^Vrvs[VrvsPtr+3]);
        SPI3BUF = d1|d2|d3|d4;
        VPtr+=4; VrvsPtr +=4;
        VPtr &= 0xD7FF; // memory address rolling
        DotCount++;					// hbgJE^XV
       
    }else{
        SPI3BUF = 0x00000000;   // SDO3̏o̓smLowɂ邽߂
        //SPI3BUF = 0x00000000;   // zero o͂B
        IEC0CLR = _IEC0_SPI3TXIE_MASK;  // SPIM荞݂֎~
    }

    IFS0CLR = _IFS0_SPI3TXIF_MASK;	// 荞݃tONA
}

/*** end of video_vga.c ****************************************************/
