/*******************************************************************

	unixio.c

	low level file functions, use Fat File System

	supported functions
		open(), close(), read(), write(), lseek(), unlink()
		gettimeofday()
	unsupported function
		link()

	(c) 2013.7.24 Suwa   http://www.suwa-koubou.jp
*******************************************************************/

#define _KERNEL

#include <stdio.h>
#include <unixio.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>
#include <sys/time.h>


#include "ff.h"
#include "rtc.h"
#include "uart2.h"

/*------------------------------------------------------------------
	global variable
------------------------------------------------------------------*/

#define	MAX_OPEN	8	// stdin(0), stdout(1), stderr(2), other from 3 to 7

static FATFS fatfs;
static FIL	 fs[MAX_OPEN];
static int   openflg[MAX_OPEN] = { 1, 1, 1, };
static int   mountflg;

void _mon_putc(char c)
{
	if(c == '\n') UART2PutChar('\r');
	UART2PutChar(c);
}

int  _mon_getc(int canblock)
{
	if(canblock){
		while(!UART2IsPressed());
		return UART2GetChar();
	}else{
		if(UART2IsPressed()){
			return UART2GetChar();
		}else{
			return -1;
		}
	}
}



/*==================================================================
	int	open(const char *pathname, int flags, int mode)
==================================================================*/

int	open(const char *pathname, unsigned int flags, unsigned int mode)
{
	int      fd;
	BYTE     md;
	FRESULT  fres;

	for(fd = 3; fd < MAX_OPEN; fd++){
		if(!openflg[fd]){
			openflg[fd] = 1;
			break;
		}
	}
	if(fd >= MAX_OPEN){
		errno = EMFILE; // Too many open files
		return -1;
	}

	// check read write flag
	if((flags & O_ACCMODE) == 0){ // O_RDONLY
		md = FA_READ;
	}else if(flags & O_WRONLY){
		md = FA_WRITE;
	}else if(flags & O_RDWR){
		md = (FA_READ|FA_WRITE);
	}else{
		errno = EINVAL;
		openflg[fd] = 0;
		return -1;
	}

	// check creat flag
	if((flags & (O_CREAT|O_EXCL)) == (O_CREAT|O_EXCL)){
		md |= FA_CREATE_NEW;
	}else if(flags & O_CREAT){
		md |= FA_CREATE_ALWAYS;
	}else if(flags | O_APPEND){
		md |= FA_OPEN_ALWAYS;
	}else{
		md |= FA_OPEN_EXISTING;
	}

	// disk mount
	if(!mountflg){
		f_mount(0, &fatfs);
		mountflg = 1;
	}else{
		mountflg++;
	}

	// file open
	fres = f_open(&(fs[fd]), (const TCHAR *)pathname, md);
	if(fres == FR_OK){
		if(flags & O_APPEND){
			f_lseek(&(fs[fd]), f_size(&(fs[fd])));
		}
		errno = 0;
		return fd;
	}

	// release FIL object
	openflg[fd] = 0;

	// disk unmount
	mountflg--;
	if(!mountflg){
		f_mount(0, NULL);
	}

	// set error cause
	switch(fres){
	case FR_NO_FILE:
	case FR_NO_PATH:
		errno = ENOENT;
		break;
	case FR_EXIST:
		errno = EEXIST;
		break;
	case FR_WRITE_PROTECTED:
		errno = EROFS;
		break;
	default:
		errno = EIO;
		break;
	}

	// return fail
	return -1;

}

/*==================================================================
	int creat(char *pathname, int mode)
==================================================================*/

int creat(const char *pathname, int mode)
{
	return open(pathname, O_CREAT|O_WRONLY|O_TRUNC, mode);
}


/*==================================================================
	int	read(int fd, void *buf, size_t nbytes)
==================================================================*/

size_t	read(int fd, void *buf, size_t nbytes)
{
	FRESULT fres;
	UINT cnt;

	// stdin ?
	if(fd == 0){
		for(cnt = 0; cnt < nbytes; cnt++){
			*(char *)buf++ = _mon_getc(1);
		}
		return nbytes;
	}


	if(fd < 0 || fd >= MAX_OPEN || !openflg[fd]){
		errno = EBADF;	// Bad file number
		return -1;
	}

	fres = f_read(&(fs[fd]), buf, nbytes, &cnt);
	if(fres == FR_OK){
		return (int)cnt;
	}else{
		errno = EIO;
		return -1;
	}
}


/*==================================================================
	int	write(int fd, void *buf, size_t nbytes)
==================================================================*/

size_t	write(int fd, const void *buf, size_t nbytes)
{
	FRESULT fres;
	UINT cnt;

	// stdout, stderr ?
	if(fd == 1 || fd == 2){
		for(cnt = 0; cnt < nbytes; cnt++){
			_mon_putc(*(char *)buf++);
		}
		return nbytes;
	}

	// other file stream 
	if(fd < 0 || fd >= MAX_OPEN || !openflg[fd]){
		errno = EBADF;	// Bad file number
		return -1;
	}

	fres = f_write(&(fs[fd]), buf, nbytes, &cnt);
	if(fres == FR_OK){
		return (int)cnt;
	}else{
		errno = EIO;
		return -1;
	}
}


/*==================================================================
	long lseek(int fd, long offset, int whence )
==================================================================*/

long	lseek(int fd, long offset, int whence)
{
	FRESULT fres;
	long	ofs;

	// file stream only!
	if(fd < 3 || fd >= MAX_OPEN || !openflg[fd]){
		errno = EBADF;	// Bad file number
		return -1;
	}

	ofs = 0;
	switch(whence){
	case SEEK_SET:
		ofs = offset;
		break;
	case SEEK_CUR:
		ofs = (long)f_tell(&(fs[fd])) + offset;
		break;
	case SEEK_END:
		ofs = (long)f_size(&(fs[fd])) + offset;
		break;
	}

	fres = f_lseek(&(fs[fd]), (DWORD)ofs);
	if(fres == FR_OK){
		return f_tell(&(fs[fd]));
	}else{
		errno = EIO;
		return -1;
	}
}


/*==================================================================
	int unlink(const char *pathname)
==================================================================*/

int unlink(const char *pathname)
{
	FRESULT fres;


	fres = f_unlink((const TCHAR *)pathname);
	if(fres == FR_OK) return 0;

	if(fres == FR_NO_FILE || fres == FR_NO_PATH){
		errno = ENOENT;
	}else{
		errno = EIO;
	}

	return -1;
}


/*==================================================================
	int	close(int fd)
==================================================================*/

int close(int fd)
{
	// can't close standard I/O(stdin(0), stdout(1), stderror(2))
	if(fd < 3 || fd >= MAX_OPEN){
		errno = EBADF;	// Bad file number
		return -1;
	}

	if(openflg[fd]){
		f_close(&(fs[fd]));
		openflg[fd] = 0;
		mountflg--;
	}

	if(!mountflg){
		f_mount(0, NULL);
	}

	return 0;
}


/*==================================================================
	int gettimeofday(struct timeval *, void *)
	seconds since 00:00:00 1day this month this year
==================================================================*/

int gettimeofday(struct timeval *tv, void *tz)
{
	char td[7];

	get_rtc(td);
	tv->tv_sec = (long)td[2] * 60 * 60 * 24 + // day
                 (long)td[3] * 60 * 60 +      // hour
                 (long)td[4] * 60 +           // minutes
                 (long)td[5];                 // second
	tv->tv_usec = 0;
	return 0;
}


/*==================================================================
	time_t	time(time_t *)
==================================================================*/

time_t	time(time_t *t)
{
	struct timeval tv;

	gettimeofday(&tv, NULL);
	if(t) *t = tv.tv_sec;
	return tv.tv_sec;
}


/*** end of unixio.c **********************************************/
