/***********************************************************************

	rtc.c
	real time clock, and timer module

	2013/10/30  S.Suwa http://www.suwa-koubou.jp	
***********************************************************************/

#include "GenericTypeDefs.h"
#include "HardwareProfile.h"
#include <stdio.h>
#include <stdlib.h>

#include "rtc.h"
#include "mz1200.h"
#include "z80.h"

void	rtc_init(void);
void	rtc_set(char *date_time);
void	get_rtcstr(char *buf);
void	get_rtc(char data[]);
DWORD	get_fattime (void);
void    delay_ms(int t);

static	volatile DWORD  timer_tick;
static	volatile WORD   year;
static	volatile BYTE	leap, month, day, hour, minute, second;
static	const    BYTE   mdays[] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31,}; 
static  volatile DWORD  delay_timer;

volatile DWORD	AplTimer;           // 1msec interval down counter for Application
volatile int    CursorBlink;        // 500msec toggle, 1Hz blink 
volatile int    KeyPolling;         // USB keyboard read interval 
volatile unsigned short ClockCount; // 1sec interval countdown timer
volatile int    tempo_strobe;       // 5msec toggle, 100Hz music tempo


/*----------------------------------------------------------------------
	initialize timer module
----------------------------------------------------------------------*/

void rtc_init(void)
{
	IEC0SET = _IEC0_T1IE_MASK; // enable T1 interrupt
	IFS0CLR = _IFS0_T1IF_MASK; // clear interrupt flag

	IPC1bits.T1IP = 2;

	//--- set interval@1ms timer
	T1CON = 0x0010;	// timer(T1), set prescale 1:8
	TMR1 = 0;
	PR1 = GetPeripheralClock()/8/1000L;	// period 1msec 
	T1CONSET = _T1CON_TON_MASK;	// start T1

	rtc_set("2013/07/16 09:00:00");
}


/*----------------------------------------------------------------------
	timer set
	rtc_set("2013/07/16 09:00:00");
----------------------------------------------------------------------*/

void	rtc_set(char *date_time)
{
	int	y, m,d, hh,mm,ss;

	y = atoi(date_time);
	m = atoi(date_time+5);
	d = atoi(date_time+8);
	hh = atoi(date_time+11);
	mm = atoi(date_time+14); 
	ss = atoi(date_time+17);

	leap = (y % 4 == 0) && (y % 100 != 0 || y % 400 == 0);

	lock_timer(); // disable T1 interrupt
	year = y; month = m; day = d;
	hour = hh; minute = mm; second = ss;
	unlock_timer(); // enable T1 interrupt

}

/*----------------------------------------------------------------------
	get_rtcstr(buf)
	 write to buf, "2012/09/01 07:52:05"
----------------------------------------------------------------------*/

void	get_rtcstr(char *buf)
{
	lock_timer(); // disable T1 interrupt
	sprintf(buf, "%4d/%02d/%02d %02d:%02d:%02d",
		year, month, day, hour, minute, second);
	unlock_timer(); // enable T1 interrupt
}

/*----------------------------------------------------------------------
	get_rtc(data[])
	 write time data to data[] array
----------------------------------------------------------------------*/

void	get_rtc(char data[])
{
	lock_timer(); // disable T1 interrupt
	data[0] = year -2000;
	data[1] = month;
	data[2] = day;
	data[3] = hour;
	data[4] = minute;
	data[5] = second;
	unlock_timer(); // enable T1 interrupt
}

/*----------------------------------------------------------------------
	Timer T1 1msec interval timer.
----------------------------------------------------------------------*/

void __attribute__ (( interrupt(ipl2), vector(_TIMER_1_VECTOR) )) _T1Interrupt(void)
{
	extern void disk_timerproc(void);


	// *** down count, etc timer *******************
	// provide timer
	if(AplTimer) AplTimer--;
	if(delay_timer) delay_timer--;

	// disk i/o command timeout timer
	disk_timerproc();

	// support Display refresh timing
	//if(RefreshTime) RefreshTime--;

	// support USB keyboard polling(10msec)
	if(!KeyPolling){
		keyboard_scan();
		KeyPolling=10;
	}else{
		KeyPolling--;
	}

	// *** toggle interval timer *************
	// update 100HZ music tempo timer  
	if(!(timer_tick % 5)) tempo_strobe ^=1;

	// update 2Hz cursor blink timer
	if(!(timer_tick % 500)) CursorBlink ^= 0x40;

	// count up timer_tick
	++timer_tick;

	// *** 1sec interval process *******
	if(timer_tick >= 1000){
	  timer_tick = 0;

	// count down clock (8253 counter#2) 
	// when counter value was reached zero, it apear interrupt.  
	  if(ClockCount){
		ClockCount--;
	  }else if(setClock){
		ClockCount = 0xA8C0;  // reload 12H*3600sec
		cpu_int = 1;
		cpu_intn = 7;
	  }

	// update real time clock
	  if(++second >= 60){
	    second = 0;
	    if(++minute >= 60){
	      minute = 0;
	      if(++hour >= 24){
	        hour = 0;
	        if(++day > (month == 2 && leap ? 29: mdays[month-1]) ){         
	          day = 1;
	          if(++month > 12){
	            month = 1;
	            year++;
				leap = (year % 4 == 0) && (year % 100 != 0 || year % 400 == 0);
	          } // month
	        } // day
	      } // hour
	    } // minute
	  } // second

	} // tick  

	// clear interrupt flag
	clear_timer();


}

/*-----------------------------------------------------------
	delay_ms()
-----------------------------------------------------------*/

void delay_ms(int t)
{
	delay_timer = (DWORD)t;
	while(delay_timer);
}


/*---------------------------------------------------------*/
/* User Provided RTC Function for FatFs module             */
/*---------------------------------------------------------*/
/* This is a real time clock service to be called from     */
/* FatFs module. Any valid time must be returned even if   */
/* the system does not support an RTC.                     */
/* This function is not required in read-only cfg.         */

DWORD get_fattime (void)
{
	DWORD tmr;

	lock_timer(); // disable T1 interrupt
	/* Pack date and time into a DWORD variable */
	tmr =	  (((DWORD)year - 1980) << 25)
			| ((DWORD)month << 21)
			| ((DWORD)day << 16)
			| (WORD)(hour << 11)
			| (WORD)(minute << 5)
			| (WORD)(second >> 1);
	unlock_timer(); // enable T1 interrupt

	return tmr;
}


/*** end of rtc.c ****************************************************/
