/***************************************************************************************

	mzfile.c

	emulate ?RDI ?RDD ?WRI ?WRD

	READ from TAPE, WRITE to TAPE emulation

	2013/10/30  S.Suwa http://www.suwa-koubou.jp	
	2014/4/12   Support .mzt file

***************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include "GenericTypeDefs.h"
#include "mz1200.h"
#include "ff.h"


#define MAX_FILE  128

static  FATFS fatfs;
static	FILINFO Finfo;
static  char  lfn_buff[64];

char *inf_file[MAX_FILE];
int   inf_max;
int   inf_next, inf_now;

/*=====================================================================================
	code convert,   from MZ ascii to standard ascii
=====================================================================================*/

const unsigned char toAscii[] = {
//         92  93  94  95  96  97   
           'e','`','~',' ','t','g',
// 98  99  9A  9B  9C  9D  9E  9F
   'h',' ','b','x','d','r','p','c',
// A0  A1  A2  A3  A4  A5  A6  A7
   'q','a','z','w','s','u','i',' ',
// A8  A9  AA  AB  AC  AD  AE  AF
   ' ','k','f','v',' ',' ',' ','j',
// B0  B1  B2  B3  B4  B5  B6  B7
   'n',' ',' ','m',' ',' ',' ','o',
// B8  B9  BA  BB  BC  BD  BE  
   ' ',' ',' ',' ',' ','y','{',
};


unsigned char mzasc2asc(unsigned char c)
{
	if(machine_mode == MD_MZ80A){
		if(c <= 0x5d) return c;
		if(c >= 0x92 && c <= 0xBE) return toAscii[c - 0x92];
		if(c == 0x8B) return '^';
		if(c == 0x80) return '}';
		if(c == 0xC0) return '|';
		return ' ';
	}

	if(machine_mode == MD_MZ1200){
		if(c <= 0x5d) return c; // numeric, alphabet
		if(c >= 0x80 && c <= 0xBF) return c + 0x20;	// KANA
		if(c == 0xe2) return '|';
		if(c == 0xc4) return '_';
		return ' ';
	}
}


/*=====================================================================================
	load ROM file
=====================================================================================*/

int load_rom(char *fname, char *buf, int size)
{
	int	in;

	printf("load_rom.\n");

	if((in = open(fname, O_RDONLY, 0)) < 0){
  		return -1;
	}

	if(read(in, buf, size) != size){
		close(in);
		return -1;
	}
	close(in);
	printf("complete load rom.\n");
	return 0;
}


/*=====================================================================================
	make MZF-file list
	2014/4/12 support mzt file
=====================================================================================*/

void	mkmzflist(void)
{
	DIR dirs;
	char *fn, *p;

	printf("Making mzf file list.\n");

	// mount disk
	f_mount(0, &fatfs);

	// set LFN buffer
	Finfo.lfname=lfn_buff;
	Finfo.lfsize=sizeof(lfn_buff);

	// search .mzf or .mzt file in root directory  2014/4/12
	if (f_opendir(&dirs, "/") == FR_OK) {
		while(f_readdir(&dirs, &Finfo) == FR_OK && Finfo.fname[0]){
			if (_FS_RPATH && Finfo.fname[0] == '.') continue;

            #if _USE_LFN
			fn = *Finfo.lfname ? Finfo.lfname : Finfo.fname;
            #else
			fn = Finfo.fname;
            #endif

			// check file extension, found .mzf or .mzt file ?
			if(strcmp(fn + (strlen(fn) -4), ".mzf") == 0 ||
			   strcmp(fn + (strlen(fn) -4), ".mzt") == 0){  // 2014/4/12

				if((p = (char *)malloc(strlen(fn)+1)) == NULL){
					printf("Can't alloc memory for file list.\n");
					break;
				}
				strcpy(p, fn);
				inf_file[inf_max] = p;
				if(++inf_max >= MAX_FILE) break;
				printf("%2d: %s\n", inf_max-1, p);

			} // if mzf || mzt
		} // while readdir
		printf("complete. make file list.\n");
	}else{
		printf("error! make file list.\n");
	}

	// un mount disk
	f_mount(0, NULL);

}

/*=====================================================================================
	cassete tape read write emulation
	SD-Card file handling
=====================================================================================*/

/*--------------------------------------------------------------------------------------
	READ TAPE INFORMATION
--------------------------------------------------------------------------------------*/

int	read_inf(void)
{
	int	in;
	char 	buf[128];

	printf("read_inf()\n");

	if(file_select() == -1) return -1;

	for(; inf_next < inf_max; inf_next++){

		if((in = open(inf_file[inf_next], O_RDONLY, 0)) < 0){
			printf("%s can't open.\n", inf_file[inf_next]);		
			continue;		
		}

		if( read(in, buf, 128) != 128 ){
			printf("read error.\n");
			close(in);
		}else{
			close(in);
			break;
		}
	}

	if(inf_next >= inf_max){
		inf_next = 0;
		return -1;
	}
	
	memcpy(RAM + 0x10f0, buf, 128);
	inf_now = inf_next;	// save now filename index
	if(++inf_next >= inf_max) inf_next = 0;
	printf("complete read information.\n");
	return 0;
}


/*--------------------------------------------------------------------------------------
	READ TAPE DATA
--------------------------------------------------------------------------------------*/

int	read_data(void)
{
	int	i, start, len, in;
	unsigned char	name[128], *p;

	printf("read_data()\n");

	play_button=0; // set motor off

	if((in = open(inf_file[inf_now], O_RDONLY, 0)) < 0)
  		return -1;

	p = RAM + 0x1102;
	len = p[0] + 256 * p[1];
	start = p[2] + 256 * p[3];

	lseek(in, 0x80, 0L);    
	if(read(in, RAM + start, len) != len){
		close(in);
		return -1;
	}

	close(in);
	printf("complete read data.\n");
	return 0;
}


/*--------------------------------------------------------------------------------------
	WARITE TAPE INFORMATION
--------------------------------------------------------------------------------------*/

int	write_inf(void)
{
	int	i, out;
	char 	buf[128];
	char	fname[128], *p;
	
	memcpy(buf, RAM + 0x10f0, 128);
	memcpy(fname, &buf[1], 16);
	fname[16] = '\0';
	if((p = strchr(fname, 0x0d)) != NULL) *p = '\0';

	// code convert from MZ ascii to standard ascii code
	for(i = 0; i < strlen(fname); i++)
		fname[i] = mzasc2asc(fname[i]);
	strcat(fname, ".mzf");

	if((out = open(fname, O_WRONLY | O_CREAT | O_TRUNC, 0)) < 0){
		printf("%s can't write/creat open.\n", fname);
		return -1;
	}

	if( write(out, buf, 128) != 128 ){
		close(out);
		unlink(fname);
		printf("write error.\n");
		return -1;
	}

	close(out);


	// update mzf file list
	// same filename exists ?
	for(i = 0; i < inf_max; i++){
		if( strcmp(inf_file[i], fname) == 0)
			return 0;
	}

	// too many files?
	if(inf_max >= MAX_FILE -1) return 0;

	if((p = (char *)malloc(strlen(fname) +1)) != NULL){
		strcpy(p, fname);
		inf_file[inf_max] = p;
		inf_max++;
	}
	return 0;
}


/*--------------------------------------------------------------------------------------
	WRITE TAPE DATA
--------------------------------------------------------------------------------------*/
		
int	write_data(void)
{
	int	i, out, start, len;
	unsigned char 	buf[128];
	char	fname[128], *p;
	
	memcpy(buf, RAM + 0x10f0, 128);

	memcpy(fname, &buf[1], 16);
	fname[16] = '\0';
	if((p = strchr(fname, 0x0d)) != NULL) *p = '\0';

	// code convert from MZ ascii to standard ascii code
	for(i = 0; i < strlen(fname); i++)
		fname[i] = mzasc2asc(fname[i]);
	strcat(fname, ".mzf");

	if((out = open(fname, O_WRONLY | O_APPEND, 0)) < 0){
		printf("%s can't append open.\n", fname);
		return -1;
	}

	len = buf[18] + buf[19] * 256;
	start = buf[20] + buf[21] * 256;

 
	if( write( out, RAM + start, len) != len ){
		printf("data write error.\n");
		close(out);
		unlink(fname);
		return -1;
	}

	close(out);
	return 0;
}


/*** end of mzfile.c ******************************************************************/


