/**********************************************************************************************

	mz1200.c

	MZ-1200/MZ-80A hardware emulation

	2013/10/31  S.Suwa http://www.suwa-koubou.jp	

    2022/7/18   o[X\,tOscrnRvrs̐䂾łȂ,
                \Ƀo[Xݒ肵āAʂ̔]\
                䂷悤ɂB
**********************************************************************************************/

#include <string.h>
#include "GenericTypeDefs.h"
#include "HardwareProfile.h"
#include "mz1200.h"
#include "video.h"
#include "mzfile.h"


/*------------------------------------------------------------------------------
	global variable
------------------------------------------------------------------------------*/

extern unsigned char keymatrix[10];	// key.c
extern volatile int  CursorBlink;	// rtc.c
extern volatile unsigned short ClockCount; // rtc.c
extern volatile int  tempo_strobe;  // rtc.c

static int           ramSwap;   // RAM swap flag
unsigned char        v_blank;   // vertical blanking
static unsigned int  keyscan;	// keybord scan row index

static int            low0,   low1,   low2;   // 8253 counter value set order flag
static unsigned short count0, count1, count2; // 8253 counter0,1,2 set value

int           machine_mode; // machine type MZ-80A or MZ-1200
int           setClock;     // ClockCount set flag
int           scrnRvrs;     // screen reverse flag
int           v_gate;       // screen blank flag
int           play_button;	// cassette deck play button 0x10=ON, 0x00=OFF (D4)

unsigned char RAM[64*1024]; // 64Kbyte RAM (0000-FFFF)
unsigned char cgROM[8][256]; // 2Kbyte character ROM
unsigned char Vrvs[1000];    // VRAM reverse flg 40x25=1000  2022/07

/*------------------------------------------------------------------------------
	function prototype
------------------------------------------------------------------------------*/

void          resetMZ1200(void);
unsigned char fetch(unsigned short adr);
void	      store(unsigned short adr, unsigned char  val);
void	      store2b(unsigned short adr, unsigned char  vh, unsigned char  vl);
unsigned char ramRead(unsigned short addr);
void	      ramWrite(unsigned short addr, unsigned char data);
unsigned char in(unsigned char dummy, unsigned short adr);
unsigned int  out(unsigned char dummy, unsigned char adr, unsigned char val);


/*---------------------------------------------------------------------------------------------
	MZ-1200 initialize (reset)
---------------------------------------------------------------------------------------------*/

void  resetMZ1200(void)
{
	int	n;

	// load ROM
	if(machine_mode == MD_MZ80A){

		// monitor
		load_rom("SA-1510.ROM", &RAM[0], 4096);
		// character
		load_rom("SA-CG.ROM", (char *)cgROM, 2048);

		// patch tape read write emulation
		//RAM[0x04CF]=0xed; RAM[0x04D0]=0xfc;	// ?RDI
		RAM[0x04E3]=0xed; RAM[0x04E4]=0xfc;	// ?RDI(CALL TMARK)
		RAM[0x04EF]=0xed; RAM[0x04F0]=0xfd;	// ?RDD
		RAM[0x0436]=0xed; RAM[0x0437]=0xfe;	// ?WRI
		RAM[0x0470]=0xed; RAM[0x0471]=0xff;	// ?WRD

		RAM[0x06B0]=0x01; // MOT2: LD B,A6H --> LD B,1  cut 2sec delay

	}else{

		// monitor
		load_rom("SP-1002.ROM", &RAM[0], 4096);
		// character
		load_rom("SP-CG.ROM", (char *)cgROM, 2048);

		// patch tape read write emulation
		//RAM[0x04D8]=0xed; RAM[0x04D9]=0xfc;	// ?RDI
		RAM[0x04EC]=0xed; RAM[0x04ED]=0xfc;	// ?RDI(CALL TMARK)
		RAM[0x04F8]=0xed; RAM[0x04F9]=0xfd;	// ?RDD
		RAM[0x0436]=0xed; RAM[0x0437]=0xfe;	// ?WRI
		RAM[0x0475]=0xed; RAM[0x0476]=0xff;	// ?WRD
	}

	// normal memory setting
	ramSwap = 0;

	// screen control
	v_blank = 1; 
	v_gate = 1;	// screen on, =0 screen off
	scrnRvrs = 0; // normal screen

	// cassette
	play_button = 0;

	// clear keymatrix
	clear_keymatrix();

	// 8253 counter set
	/***
	counter #0  for sound generate, counter#0 value is nn'.
	    original clock 2MHz, output frequency 2MHz/nn'/2,
         Sound frequency is divided 2 by outside audio circuit.

       simulate by Timer3
        clock 80MHz, prescale 32 = 80M/32=2.5Mhz
        interrupt interval 2.5MHz/(nn'*1.25), out put frequency 2.5MHz/(nn'*1.25)
        TMR3=0,PR3=(nn'+nn'/4)-1
        toggle digital outputpin by interrput routine
	counter #1, 1 sec interval timer
	counter #2, Time Clock, 1sec countdown, at reached RST 38H
	***/
	T3CON = 0x8050;	// prescale 1:32
	TMR3 = 0;
	PR3 = 0;
	IEC0CLR = _IEC0_T3IE_MASK; // disable interrupt
	IFS0CLR = _IFS0_T3IF_MASK; // flag clear
	IPC3bits.T3IP = 2;         //@priority 2
	TRISBbits.TRISB9 = 0;	   // RB9 as sound out

	// make file list
	mkmzflist();

	// cpu reset
	z80reset();
}

/*---------------------------------------------------------------------------------------------
	generate music frequency by T3 interrupt
---------------------------------------------------------------------------------------------*/

void __attribute__ (( interrupt(ipl2), vector(_TIMER_3_VECTOR) )) _T3Interrupt(void)
{
	IFS0CLR=_IFS0_T3IF_MASK;
	LATBINV = 0x0200;	// toggle RB9
}


/*---------------------------------------------------------------------------------------------
	data fetch from RAM
---------------------------------------------------------------------------------------------*/

unsigned char fetch(unsigned short adr)
{
	if((adr & 0xFD00)==0xE000) return ramRead(adr);  // E0xx or E2xx
	return RAM[adr];
}

/*---------------------------------------------------------------------------------------------
	data store to RAM
---------------------------------------------------------------------------------------------*/

// 8bit memory write
void    store(unsigned short adr, unsigned char val) // 8bit memory write
{
	if((adr & 0xff00) == 0xe000) ramWrite(adr, val);
	else RAM[adr] = val;
}
                               
// 16bit memory write
void    store2b(unsigned short adr, unsigned char vh, unsigned char vl)
{
	store(adr, vl);
	store(adr+1, vh);
}


/*---------------------------------------------------------------------------------------------
	ramRead(addr)
---------------------------------------------------------------------------------------------*/

unsigned char ramRead(unsigned short addr)
{
	int n,t;

	switch(addr){

  	case 0xe001:
		return keymatrix[keyscan];

  	case 0xe002:
		/* bit 7  V_BLANK 
		 * bit 6  cursor blink    
		 * bit 5  read data from casset
		 * bit 4  casset play button
		 */		
		return v_blank | CursorBlink | play_button;

	case 0xe005: // read 8253 counter#1 value
		low1 ^= 1;
		if(!low1){
			return count1;
		}else{
			return (count1 >> 8);
		}

  	case 0xe006: // read 8253 counter#2 value
		low2 ^= 1;
		if(!low2){
			count2 = ClockCount;
			return count2;
		}else{
			return (count2 >> 8);
		}

  	case 0xe008: // tempo frequency is 100Hz(5ms toggle)
    	return tempo_strobe;

	case 0xe00c: // RAM memory swap on, 0000-0FFF <-> C000-CFFF
		if(!ramSwap){
			ramSwap = 1;
			for(n = 0; n < 4096; n++){
				t = RAM[n];
				RAM[n] = RAM[0xc000+n];
				RAM[0xc000+n] = t;
			}
		}
		break;

	case 0xe010: // RAM memory swap off
		if(ramSwap){
			ramSwap = 0;
			for(n = 0; n < 4096; n++){
				t = RAM[n];
				RAM[n] = RAM[0xc000+n];
				RAM[0xc000+n] = t;
			}
		}
		break;

	case 0xe014:
		scrnRvrs = 0; // reset to screen reverse
        memset(Vrvs, 0x00, sizeof(Vrvs)); // 2022/07
		break;

	case 0xe015:
		scrnRvrs = 1; // set to screen reverse
        memset(Vrvs, 0xFF, sizeof(Vrvs)); // 2022/07
		break;

  	default:
		// screen hardware roll up, roll down (MZ-80A)
		if(addr >= 0xE200 && addr <= 0xE2FF){
			VMem = 0xD000 + (addr & 0xFF) * 8;
		}
		break;
  	}

	return 0xff;
}


/*---------------------------------------------------------------------------------------------
	ramWrite(addr, val)
---------------------------------------------------------------------------------------------*/

void	ramWrite(unsigned short addr, unsigned char data)
{
	switch(addr){
  	case 0xe000: // bit 0-3 keyboard scan row out
   		keyscan = data & 0x0f;
 		break;
    
  	case 0xe002:
		/* bit 3  casset motor on/off
		 * bit 2  N.C
		 * bit 1  write data to casset    
		 * bit 0  V_GATE 
		 */		
		v_gate = !(data & 1);
		break;
  
  	case 0xe003: // 8255 mode set, port C single bit set/reset, all ignore
		break;

  	case 0xe004: // sound freq stuff written here
		if(low0){
			count0 = data;
		}else{
			count0 |= ((unsigned short)data << 8);
			PR3=(count0+count0/4)-1;
		}
		low0 ^= 1;
		break;

  	case 0xe005: // set counter#1 value
		// counter 1, 1sec interval timer
		if(low1){
			count1 = data;
		}else{
			count1 |= ((unsigned short)data << 8);
		}
		low1 ^= 1;
		break;

  	case 0xe006: // set counter#2 value
		// cont2 - this takes the countdown timer */
		if(low2){
			count2 = data;
		}else{
			count2 |= ((unsigned short)data << 8);
			ClockCount = count2;
			setClock = 1;
		}
		low2 ^= 1;
		break;
 
  	case 0xe007: // 8253 mode set
		switch(data&0xC0){
		case 0x00: // counter #0
			low0=(data&0x30 == 0x10)?0:1;
			break;

		case 0x40: // counter #1
			low1=(data&0x30 == 0x10)?0:1;
			break;

		case 0x80: // counter #2
			low2=(data&0x30 == 0x10)?0:1;
			break;
		default:
			break;
		}
		break;

  	case 0xe008: // bit 0 controls whether sound is on or off
		if(data & 1){ // music start(Timer3)
			IFS0CLR = _IFS0_T3IF_MASK;
			IEC0SET = _IEC0_T3IE_MASK; // enable interrupt
		}else{ // music stop
			IFS0CLR = _IFS0_T3IF_MASK;
			IEC0CLR = _IEC0_T3IE_MASK; // disable interrupt
			LATBCLR=0x0200;  // RB9 to low
		}
		break;

	default:
		break;
  	}
}


/*---------------------------------------------------------------------------------------------
	IN(), OUT() emulation (dummy)
---------------------------------------------------------------------------------------------*/

unsigned char	in(unsigned char dummy, unsigned short adr)
{
	return (0xff);
}

unsigned int	out(unsigned char dummy, unsigned char adr, unsigned char val)
{
	return (0);
}

/*** end of mz1200.c *************************************************************************/
