/*******************************************************************************

    main.c

	MZ-80 (MZ-80A/MZ-1200) emulator for PIC32MX board

	2013/11/1  S.Suwa http://www.suwa-koubou.jp	

    2022/7/21 update to Support VGA display

 specification
	PIC           PIC32MX695F512H
	Memory disk   SDCard, store mzf(mzt) file
	Display       VGA display
    Keyboard      USB 106 japanese key
@@@@Speaker       small speaker

    PIC32MX695F512H pin assign

      VGA Signal
        RD10       --- VGA V-Sync
        RD0(OC1)   --- VGA H-Sync
        RD3(SDO3)  --- VGA Green video signal

      SOUND
        RB9        --- small Speaker

      UART2
	    U2TX(RF5)  --- debug print to PC
	    U2RX(RF4)  --- debug print to PC

      MMC
        RG6/SCK2   --- MMC_SCK
        RG7/SDI2   --- MMC_SDO
        RG8/SDO2   --- MMC_SDI
        RG9        --- MMC_CS 
        RB5(PU)    --- MMC_CD 

      INTERRUPT PRIORITY LEVEL
        ipl2   _T1Interrupt   1msec interval timer
        ipl2   _T3Interrupt   sound generater 
        ipl4   _USBInterrupt  USB event(DETACH,SOF,TRN,etc)
        ipl6   _T2Interrupt   VGA sync signal
        ipl5   _OC2Interrupt  VGA Video signal start trigger
        ipl5   _SPI3TXInterrupt VGA Video signal out

      Timer modules
        T1    1msec system timer.  
        T2    VGA sync signal generate
        T3    sound generate
        T5    Z80 cpu clock

      OC modules
        OC1   VGA horizontal-sync signal
        OC2   VGA video signal start trigger (not use pinout)

      SPI modules
        SPI2  mmc (SD Card interface control)
        SPI3  VGA video signal out

      USB module
        U1    USB keybord
  

*******************************************************************************/

#include "GenericTypeDefs.h"
#include "HardwareProfile.h"
#include "rtc.h"
#include "uart2.h"
#include "mz1200.h"
#include "key.h"
#include "z80.h"


/*------------------------------------------------------------------------------
	Configuration for PIC32MX695F 
------------------------------------------------------------------------------*/

#include "p32mx695f_config.h"


/*------------------------------------------------------------------------------
	definition
------------------------------------------------------------------------------*/


/*------------------------------------------------------------------------------
	function prototype
------------------------------------------------------------------------------*/

void   InitializeSystem(void);


/*------------------------------------------------------------------------------
	global variable
------------------------------------------------------------------------------*/


/*==============================================================================
	main
==============================================================================*/

int main ( void )
{
	unsigned short key;

    // I/O initialize 
    InitializeSystem();

	// UART2 initialize
	UART2Init();

	// keyboard(USB Host driver) initialize
	keyboard_init();

	// disable keyboard scan from rtc module
	scan_inhibit = 1; // <--- add 2014/4/8

	// RTC initialize
	rtc_init();

	// emulate MZ-80A or MZ-1200?
	//   press any keys to MZ-80A mode. check time is 1sec.
	machine_mode = MD_MZ1200; // set default mode
	AplTimer=1000;
	while(AplTimer){
		key = keyboard_read();
		if(key && key != 0xffff){
			machine_mode = MD_MZ80A;
			break;
		}
	}


	// opening message
	printf("*** Start MZ-80 Emulator.\r\n");

	// power on MZ-80
	resetMZ1200();

	// NTSC video initialize
	video_init();

	
	// enable keyboard scan from rtc module
	scan_inhibit = 0; // <--- add 2014/4/8

	// go emulate
	execZ80();
}


/*==============================================================================
	InitializeSystem() for REPIC32SD Board
==============================================================================*/

void InitializeSystem ( void )
{
	int	value;

	// Set System Config
	//SYSTEMConfigPerformance(GetSystemClock());
	value = SYSTEMConfigWaitStatesAndPB( GetSystemClock() );

	// Enable the cache for the best performance	
	CheKseg0CacheOn();

	// enable multi vector
	INTEnableSystemMultiVectoredInt();

	value = OSCCON;
	while (!(value & 0x00000020)){
		value = OSCCON;    // Wait for PLL lock to stabilize	
	}

	INTEnableInterrupts();

	// ADC input off
	AD1PCFG = 0xffff;
	// disable JTAG
	DDPCONbits.JTAGEN = 0;

	//--- setting I/O pins               
    TRISB=0x0020;
    LATB=0x3000;
                // RB5        input,  MMC_CD
                // RB9        output, sound out, initial low
				// RB12,RB13  output, LED monitor, initial high
                // others     output, output level low

	TRISC = 0x0000;
	LATC = 0x0000;
                // RC all pins not use

	LATD=	0x0000;
	TRISD=	0x0004;
                // RD0(OC1)      output  H-Sync
                // RD1(SCK3/OC2) output, N.C
                // RD2(SDI3/OC3) input,  N.C
                // RD3(SDO3)     output, Green video signal
                // RD10          output, V-Sync
                // others        output, output level low

	TRISE = 0x0000;
	LATE = 0x0000;
                // RE all pins not use

	TRISF = 0x0010;
	LATF = 0x0020;
                // RF4(U2RX) is input.
                // RF5(U2TX) to level high.

	LATG=	0x0200;
	TRISG=	0x0080;
                // RG6(SCK2)   output, MMC_SCK
                // RG7(SDI2)   input,  MMC_SDO
                // RG8(SDO2)   output, MMC_SDI
                // RG9(CS)     output, MMC_CS   initial level high

	//--- internal pullup
	CNPUEbits.CNPUE7=1;	// RB5(CN7) MMC_CD Pull-up
                        // other pins low level output, don't pull up.

} // InitializeSystem


/*** end of main.c ************************************************************/
