/***************************************************************************

    USB Keyboard handling

    Original source code is "keybord_demo.c". It's included "USB Keyboard
    Host Application Demo" in Microchip Application libraly. 
	I was modified it.

	2013/10/24  S.Suwa http://www.suwa-koubou.jp	

****************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "GenericTypeDefs.h"
//#include "HardwareProfile.h"
#include "usb_config.h"
#include "usb.h"
#include "usb_host_hid_parser.h"
#include "usb_host_hid.h"


/*---------------------------------------------------------------------------
    Data Structures
---------------------------------------------------------------------------*/

typedef enum _APP_STATE
{
    DEVICE_NOT_CONNECTED,
    DEVICE_CONNECTED, /* Device Enumerated  - Report Descriptor Parsed */
    READY_TO_TX_RX_REPORT,
    GET_INPUT_REPORT, /* perform operation on received report */
    INPUT_REPORT_PENDING,
    SEND_OUTPUT_REPORT, /* Not needed in case of mouse */
    OUTPUT_REPORT_PENDING,
    ERROR_REPORTED 
} APP_STATE;

typedef struct _HID_REPORT_BUFFER
{
    WORD  Report_ID;
    WORD  ReportSize;
    BYTE* ReportData;
    WORD  ReportPollRate;
}   HID_REPORT_BUFFER;

typedef struct _HID_LED_REPORT_BUFFER
{
    BYTE  NUM_LOCK      : 1;
    BYTE  CAPS_LOCK     : 1;
    BYTE  SCROLL_LOCK   : 1;
    BYTE  UNUSED        : 5;
}   HID_LED_REPORT_BUFFER;


/*---------------------------------------------------------------------------
    Internal Function Prototypes
---------------------------------------------------------------------------*/

BOOL USB_ApplicationEventHandler( BYTE address, USB_EVENT event, void *data, DWORD size );

void keyboard_init(void);
WORD keyboard_read(void);
WORD App_ProcessInputReport(void);
void App_PrepareOutputReport(void);
BOOL USB_HID_DataCollectionHandler(void);


/*---------------------------------------------------------------------------
    Macros
---------------------------------------------------------------------------*/

#define USAGE_PAGE_LEDS        (0x08)
#define USAGE_PAGE_KEY_CODES   (0x07)
#define USAGE_MIN_MODIFIER_KEY (0xE0)
#define USAGE_MAX_MODIFIER_KEY (0xE7)
#define USAGE_MIN_NORMAL_KEY   (0x00)
#define USAGE_MAX_NORMAL_KEY   (0xFF)
#define HID_CAPS_LOCK_VAL      (0x39)
#define HID_NUM_LOCK_VAL       (0x53)
#define MAX_ERROR_COUNTER      (10)


/*---------------------------------------------------------------------------
    Global Variables
---------------------------------------------------------------------------*/

APP_STATE App_State_Keyboard = DEVICE_NOT_CONNECTED;

HID_DATA_DETAILS Appl_LED_Indicator;
HID_DATA_DETAILS Appl_ModifierKeysDetails;
HID_DATA_DETAILS Appl_NormalKeysDetails;

HID_REPORT_BUFFER     Appl_raw_report_buffer;
HID_LED_REPORT_BUFFER Appl_led_report_buffer;

BYTE ErrorDriver;
BYTE ErrorCounter;
BYTE NumOfBytesRcvd;

BOOL ReportBufferUpdated;
BOOL LED_Key_Pressed = FALSE;
BYTE CAPS_Lock_Pressed = 0;
BYTE NUM_Lock_Pressed = 0;


/*---------------------------------------------------------------------------
    USB Support Functions
---------------------------------------------------------------------------*/

BOOL USB_ApplicationEventHandler( BYTE address, USB_EVENT event, void *data, DWORD size )
{
    switch( (INT)event )
    {
        case EVENT_VBUS_REQUEST_POWER:
        case EVENT_VBUS_RELEASE_POWER:
        case EVENT_HUB_ATTACH:
        case EVENT_UNSUPPORTED_DEVICE:
        case EVENT_CANNOT_ENUMERATE:
        case EVENT_CLIENT_INIT_ERROR:
        case EVENT_OUT_OF_MEMORY:
        case EVENT_UNSPECIFIED_ERROR:
            return TRUE;

		case EVENT_HID_ATTACH:
            #ifdef DEBUG_MODE
                UART2PrintString( "\r\nKeyboard Attached\r\n" );
			#endif
            return TRUE;
	
		case EVENT_HID_DETACH:
			App_State_Keyboard = DEVICE_NOT_CONNECTED;
            #ifdef DEBUG_MODE
                UART2PrintString( "\r\nKeyboard Detached\r\n" );
			#endif
			return TRUE;

		case EVENT_HID_RPT_DESC_PARSED:
            #ifdef DEBUG_MODE
                UART2PrintString( "\r\nKeyboard report descriptor parsed\r\n" );
			#endif
            return(USB_HID_DataCollectionHandler());

        default:
            break;
    }
    return FALSE;
}


/*=============================================================================
    void keyboard_init(void)
=============================================================================*/

void keyboard_init(void)
{
	// set state to Keyboard detached
	App_State_Keyboard = DEVICE_NOT_CONNECTED;

    // Initialize USB layers
    USBInitialize(0);

	while(!USBHostHID_ApiDeviceDetect()){
        USBTasks();	// USBHost & USBHID
	}
    App_State_Keyboard = GET_INPUT_REPORT;

}


/*=============================================================================
    BYTE keyboard_read(void)
=============================================================================*/

WORD keyboard_read(void)
{
	WORD data;

	USBTasks();

    switch(App_State_Keyboard){
	case DEVICE_NOT_CONNECTED:
		if(USBHostHID_ApiDeviceDetect()){
	        App_State_Keyboard = GET_INPUT_REPORT;
		}
	    return 0xffff;

    case GET_INPUT_REPORT:
        if(!USBHostHID_ApiGetReport(Appl_raw_report_buffer.Report_ID,Appl_ModifierKeysDetails.interfaceNum,
                                Appl_raw_report_buffer.ReportSize, Appl_raw_report_buffer.ReportData))
        {
            App_State_Keyboard = INPUT_REPORT_PENDING;
        }
        return 0xffff;

    case INPUT_REPORT_PENDING:
        if(USBHostHID_ApiTransferIsComplete(&ErrorDriver,&NumOfBytesRcvd))
        {
            if(ErrorDriver ||(NumOfBytesRcvd != Appl_raw_report_buffer.ReportSize ))
            {
                ErrorCounter++ ; 
                if(MAX_ERROR_COUNTER <= ErrorDriver)
                {
                    App_State_Keyboard = ERROR_REPORTED;
                }
                else
                {
        			App_State_Keyboard = GET_INPUT_REPORT;
                }
            }
            else
            {
                ErrorCounter = 0; 
                ReportBufferUpdated = TRUE;
       			App_State_Keyboard = GET_INPUT_REPORT;

                data = App_ProcessInputReport();
                App_PrepareOutputReport();
				
				return data; // include NULL key(NULL apea at key release)
            }
        }
        return 0xffff; 

    case SEND_OUTPUT_REPORT: /* Will be done while implementing Keyboard */
        if(!USBHostHID_ApiSendReport(Appl_LED_Indicator.reportID,Appl_LED_Indicator.interfaceNum, Appl_LED_Indicator.reportLength,
                                (BYTE*)&Appl_led_report_buffer))
        {
            App_State_Keyboard = OUTPUT_REPORT_PENDING;
        }
		return 0xffff;

    case OUTPUT_REPORT_PENDING:
         if(USBHostHID_ApiTransferIsComplete(&ErrorDriver,&NumOfBytesRcvd))
         {
             if(ErrorDriver)
             {
                 ErrorCounter++ ; 
                 if(MAX_ERROR_COUNTER <= ErrorDriver)
                 {
                     App_State_Keyboard = ERROR_REPORTED;
                 }
             }
             else
             {
                 ErrorCounter = 0; 
       			 App_State_Keyboard = GET_INPUT_REPORT;
             }
         }
         return 0xffff;

    case ERROR_REPORTED:
        return 0xffff;

    default:
        break;

    }
    return 0xffff;
}          


/*---------------------------------------------------------------------------
    void App_ProcessInputReport(void)
---------------------------------------------------------------------------*/

WORD App_ProcessInputReport(void)
{
    int  i;
	WORD data;

	// get data
	data = ((WORD)Appl_raw_report_buffer.ReportData[0] << 8) | Appl_raw_report_buffer.ReportData[2];

	// check CapsLock NumericLock
	for(i=2;i<(Appl_raw_report_buffer.ReportSize);i++) 
	{
		if(Appl_raw_report_buffer.ReportData[i] != 0)
		{
            if(Appl_raw_report_buffer.ReportData[i] == HID_CAPS_LOCK_VAL)
            {
               CAPS_Lock_Pressed = !CAPS_Lock_Pressed;
               LED_Key_Pressed = TRUE;
               Appl_led_report_buffer.CAPS_LOCK = CAPS_Lock_Pressed;

            }
            else if(Appl_raw_report_buffer.ReportData[i] == HID_NUM_LOCK_VAL)
            {
                NUM_Lock_Pressed = !NUM_Lock_Pressed;
                LED_Key_Pressed = TRUE;
                Appl_led_report_buffer.NUM_LOCK = NUM_Lock_Pressed;
            }
		}
	}

	if(CAPS_Lock_Pressed && (data & 0xff)) data |= 0x8000;
	if(NUM_Lock_Pressed && (data & 0xff)) data |= 0x0800;

	// clear receive buffer
    for(i=0;i<Appl_raw_report_buffer.ReportSize;i++)
    {
        Appl_raw_report_buffer.ReportData[i] = 0;
    }

	return data;

}


/*---------------------------------------------------------------------------
    void App_PrepareOutputReport(void)
---------------------------------------------------------------------------*/

void App_PrepareOutputReport(void)
{
    if(ReportBufferUpdated == TRUE)
    {
        ReportBufferUpdated = FALSE;
        if(LED_Key_Pressed)
        {
            App_State_Keyboard = SEND_OUTPUT_REPORT;
            LED_Key_Pressed = FALSE;
        }
     }
}


/*---------------------------------------------------------------------------
    BOOL USB_HID_DataCollectionHandler(void)
---------------------------------------------------------------------------*/

BOOL USB_HID_DataCollectionHandler(void)
{
  BYTE NumOfReportItem = 0;
  BYTE i;
  USB_HID_ITEM_LIST* pitemListPtrs;
  USB_HID_DEVICE_RPT_INFO* pDeviceRptinfo;
  HID_REPORTITEM *reportItem;
  HID_USAGEITEM *hidUsageItem;
  BYTE usageIndex;
  BYTE reportIndex;
  BOOL foundLEDIndicator = FALSE;
  BOOL foundModifierKey = FALSE;
  BOOL foundNormalKey = FALSE;

  pDeviceRptinfo = USBHostHID_GetCurrentReportInfo(); // Get current Report Info pointer
  pitemListPtrs = USBHostHID_GetItemListPointers();   // Get pointer to list of item pointers

  BOOL status = FALSE;
   /* Find Report Item Index for Modifier Keys */
   /* Once report Item is located , extract information from data structures provided by the parser */
   NumOfReportItem = pDeviceRptinfo->reportItems;
   for(i=0;i<NumOfReportItem;i++)
    {
       reportItem = &pitemListPtrs->reportItemList[i];
       if((reportItem->reportType==hidReportInput) && (reportItem->dataModes == HIDData_Variable)&&
           (reportItem->globals.usagePage==USAGE_PAGE_KEY_CODES))
        {
           /* We now know report item points to modifier keys */
           /* Now make sure usage Min & Max are as per application */
            usageIndex = reportItem->firstUsageItem;
            hidUsageItem = &pitemListPtrs->usageItemList[usageIndex];
            if((hidUsageItem->usageMinimum == USAGE_MIN_MODIFIER_KEY)
                &&(hidUsageItem->usageMaximum == USAGE_MAX_MODIFIER_KEY)) //else application cannot suuport
            {
               reportIndex = reportItem->globals.reportIndex;
               Appl_ModifierKeysDetails.reportLength = (pitemListPtrs->reportList[reportIndex].inputBits + 7)/8;
               Appl_ModifierKeysDetails.reportID = (BYTE)reportItem->globals.reportID;
               Appl_ModifierKeysDetails.bitOffset = (BYTE)reportItem->startBit;
               Appl_ModifierKeysDetails.bitLength = (BYTE)reportItem->globals.reportsize;
               Appl_ModifierKeysDetails.count=(BYTE)reportItem->globals.reportCount;
               Appl_ModifierKeysDetails.interfaceNum= USBHostHID_ApiGetCurrentInterfaceNum();
               foundModifierKey = TRUE;
            }

        }
        else if((reportItem->reportType==hidReportInput) && (reportItem->dataModes == HIDData_Array)&&
           (reportItem->globals.usagePage==USAGE_PAGE_KEY_CODES))
        {
           /* We now know report item points to modifier keys */
           /* Now make sure usage Min & Max are as per application */
            usageIndex = reportItem->firstUsageItem;
            hidUsageItem = &pitemListPtrs->usageItemList[usageIndex];
            if((hidUsageItem->usageMinimum == USAGE_MIN_NORMAL_KEY)
                &&(hidUsageItem->usageMaximum <= USAGE_MAX_NORMAL_KEY)) //else application cannot suuport
            {
               reportIndex = reportItem->globals.reportIndex;
               Appl_NormalKeysDetails.reportLength = (pitemListPtrs->reportList[reportIndex].inputBits + 7)/8;
               Appl_NormalKeysDetails.reportID = (BYTE)reportItem->globals.reportID;
               Appl_NormalKeysDetails.bitOffset = (BYTE)reportItem->startBit;
               Appl_NormalKeysDetails.bitLength = (BYTE)reportItem->globals.reportsize;
               Appl_NormalKeysDetails.count=(BYTE)reportItem->globals.reportCount;
               Appl_NormalKeysDetails.interfaceNum= USBHostHID_ApiGetCurrentInterfaceNum();
               foundNormalKey = TRUE;
            }

        }
        else if((reportItem->reportType==hidReportOutput) &&
                (reportItem->globals.usagePage==USAGE_PAGE_LEDS))
        {
            usageIndex = reportItem->firstUsageItem;
            hidUsageItem = &pitemListPtrs->usageItemList[usageIndex];

            reportIndex = reportItem->globals.reportIndex;
            Appl_LED_Indicator.reportLength = (pitemListPtrs->reportList[reportIndex].outputBits + 7)/8;
            Appl_LED_Indicator.reportID = (BYTE)reportItem->globals.reportID;
            Appl_LED_Indicator.bitOffset = (BYTE)reportItem->startBit;
            Appl_LED_Indicator.bitLength = (BYTE)reportItem->globals.reportsize;
            Appl_LED_Indicator.count=(BYTE)reportItem->globals.reportCount;
            Appl_LED_Indicator.interfaceNum= USBHostHID_ApiGetCurrentInterfaceNum();
            foundLEDIndicator = TRUE;
        }

    }


   if(pDeviceRptinfo->reports == 1)
    {
        Appl_raw_report_buffer.Report_ID = 0;
        Appl_raw_report_buffer.ReportSize = (pitemListPtrs->reportList[reportIndex].inputBits + 7)/8;
        Appl_raw_report_buffer.ReportData = (BYTE*)malloc(Appl_raw_report_buffer.ReportSize);
        Appl_raw_report_buffer.ReportPollRate = pDeviceRptinfo->reportPollingRate;
        if((foundNormalKey == TRUE)&&(foundModifierKey == TRUE))
        status = TRUE;
    }

    return(status);
}

/*** end of keyboard.c ******************************************************/
