/***************************************************************************************

	key.c
 
	Scaned USB keyboard and sets a keymatrix array

 Note:
    This program supports emulating of MZ-1200 Japanese keyboard 
    and MZ-80A Europian keyboard.

	2013/10/31  S.Suwa http://www.suwa-koubou.jp	

***************************************************************************************/
#include <string.h>
#include "GenericTypeDefs.h"
#include "key.h"
#include "mz1200.h"
#include "z80.h"
 
const BYTE code_convert[2][112]={
// MZ-80A Europian keyboard
 {
  //   0    1    2    3    4    5    6    7    8    9   Number
  //0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,0x0C,0x0D,  USB key data
  //   A    B    C    D    E    F    G    H    I    J   Character
	  11,  24,  16,  19,  20,  26,  27,  34,  37,  35,  // MZ-80 keymatrix position(row*8+col)
                                                        // need shift +128, unsupport 255
  //  10   11   12   13   14   15   16   17   18   19
  //0x0E,0x0F,0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17
  //   K    L    M    N    O    P    Q    R    S    T 
      42,  43,  40,  33,  44,  45,  12,  21,  18,  28,

  //  20   21   22   23   24   25   26   27   28   29
  //0x18,0x19,0x1A,0x1B,0x1C,0x1D,0x1E,0x1F,0x20,0x21
  //   U    V    W    X    Y    Z    1    2    3,   4
      36,  25,  13,  17,  29,   8,   14,  15, 22,  23,

  //  30   31   32   33   34   35   36   37   38   39
  //0x22,0x23,0x24,0x25,0x26,0x27,0x28,0x29,0x2A,0x2B
  //   5    6    7    8    9    0   CR  ESC   BS  TAB
      30,  31,  38,  39,  46,  47,  59, 255,  10, 255,

  //  40   41   42   43   44   45   46   47   48   49
  //0x2C,0x2D,0x2E,0x2F,0x30,0x31,0x32,0x33,0x34,0x35
  // SPC    -    ^ @@ @    [   ??    ]    ;    :  KNJ 
      32,  54,  55,  52,  53, 255,  58,  50,  51, 255,

  //  50   51   52   53   54   55   56   57   58   59
  //0x36,0x37,0x38,0x39,0x3A,0x3B,0x3C,0x3D,0x3E,0x3F
  //   ,    .    / CpsLk  F1   F2   F3   F4   F5   F6
      41,  48,  49, 255, 255, 255, 255, 255, 255, 255,

  //  60   61   62   63   64   65   66   67   68   69
  //0x40,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,
  //  F7   F8   F9  F10  F11  F12 SysRq Scrl BRK INST 
     255, 255, 255, 255, 255, 255, 255, 255, 135, 138,

  //  70   71   72   73   74   75   76   77   78   79
  //0x4A,0x4B,0x4C,0x4D,0x4E,0x4F,0x50,0x51,0x52,0x53
  //HOME PgUP  DEL  END PgDN RIGHT LEFT DOWN UP  NmLK
      63, 255,  10, 191, 255,  61, 189, 188,  60, 255,

  //  80   81   82   83   84   85   86   87   88   89
  //0x54,0x55,0x56,0x57,0x58,0x59,0x5A,0x5B,0x5C,0x5D
  //   /    *    -    +   ??    1    2    3    4    5
      49, 179,  175,178, 255,  66,  67,  74,  68,  69,

  //  90   91   92   93   94   95   96   97   98   99
  //0x5E,0x5F,0x60,0x61,0x62,0x87,0x88,0x89,0xFF,0xFF
  //   6    7    8    9    0     KANA  YEN  <NC> <NC>
      76,  70,  71,  78,  64, 255,   1,  62, 255, 255,

  // 100  101  102  103  104  105  106  107  108  109   +SHIFT
  //0x2D,0x2E,0x2F,0x30,0x31,0x32,0x33,0x34,0x35,0x36
  //  =    ~    `    {    ??   }    +    *   ??    <
    182, 183, 180,  181, 255, 186, 178, 179, 255, 169,

  // 110  111
  //0x37,0x38,
  //  >    ?
    176,  56,
 },    

// MZ-1200 Japanese keyboard
 {
  //   0    1    2    3    4    5    6    7    8    9   Number
  //0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,0x0C,0x0D,  USB key data
  //   A    B    C    D    E    F    G    H    I    J   Character
      32,  50,  49,  33,  17,  41,  34,  42,  27,  35,  // MZ-80 keymatrix position(row*8+col)
                                                        // need shift +128, unsupport 255
  //  10   11   12   13   14   15   16   17   18   19
  //0x0E,0x0F,0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17
  //   K    L    M    N    O    P    Q    R    S    T 
      43,  36,  51,  58,  20,  28,  16,  25,  40,  18,

  //  20   21   22   23   24   25   26   27   28   29
  //0x18,0x19,0x1A,0x1B,0x1C,0x1D,0x1E,0x1F,0x20,0x21
  //   U    V    W    X    Y    Z    1    2    3,   4
      19,  57,  24,  56,  26,  48,   0,   8,   1,   9,

  //  30   31   32   33   34   35   36   37   38   39
  //0x22,0x23,0x24,0x25,0x26,0x27,0x28,0x29,0x2A,0x2B
  //   5    6    7    8    9    0   CR  ESC   BS  TAB
       2,  10,   3,  11,   4,  12,  68, 255,  65, 255,

  //  40   41   42   43   44   45   46   47   48   49
  //0x2C,0x2D,0x2E,0x2F,0x30,0x31,0x32,0x33,0x34,0x35
  // SPC    -  ^+ @@@+   [   ??    ]    ;    :  KNJ 
      73,   5,  21,  77, 153, 255, 146,  44, 148,  53,

  //  50   51   52   53   54   55   56   57   58   59
  //0x36,0x37,0x38,0x39,0x3A,0x3B,0x3C,0x3D,0x3E,0x3F
  //   ,    .    / CpsLk  F1   F2   F3   F4   F5   F6
      59,  52,  60, 255, 255, 255, 255, 255, 255, 255,

  //  60   61   62   63   64   65   66   67   68   69
  //0x40,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,
  //  F7   F8   F9  F10  F11  F12 SysRq Scrl BRK INST 
     255, 255, 255, 255, 255, 255, 255, 255,  75, 193,

  //  70   71   72   73   74   75   76   77   78   79
  //0x4A,0x4B,0x4C,0x4D,0x4E,0x4F,0x50,0x51,0x52,0x53
  //HOME PgUP  DEL  END PgDN RIGHT LEFT DOWN UP  NmLK
      72, 255,  65, 200, 255,  67, 195,  74, 202, 255,

  //  80   81   82   83   84   85   86   87   88   89
  //0x54,0x55,0x56,0x57,0x58,0x59,0x5A,0x5B,0x5C,0x5D
  //   /    *    -    +   ??    1    2    3    4    5
      60, 149,   5, 133, 255,   0,   8,   1,   9,   2,

  //  90   91   92   93   94   95   96   97   98   99
  //0x5E,0x5F,0x60,0x61,0x62,0x87,0x88,0x89,0xFF,0xFF
  //   6    7    8    9    0     KANA  YEN  <NC> <NC>
      10,   3,  11,   4,  12,  37,  53,  15, 255, 255,

  // 100  101  102  103  104  105  106  107  108  109   +SHIFT
  //0x2D,0x2E,0x2F,0x30,0x31,0x32,0x33,0x34,0x35,0x36
  //  =    ~    `    {    ??   }    +    *   ??    <
      21, 255, 255, 255, 255, 255, 133, 149, 255, 144,

  // 110  111
  //0x37,0x38,
  //  >    ?
    152, 155,
 },
};    


/*---------------------------------------------------------------------------------------------
	global variable
---------------------------------------------------------------------------------------------*/

unsigned char keymatrix[10];
int scan_inhibit;


/*---------------------------------------------------------------------------------------------
	void clear_keymatrix(void)
---------------------------------------------------------------------------------------------*/

void clear_keymatrix(void)
{
	memset(keymatrix, 0xff, 10);
}	


/*---------------------------------------------------------------------------------------------
	void keyboard_scan(void)
---------------------------------------------------------------------------------------------*/

void keyboard_scan(void)
{
	WORD key;
	BYTE data;
	int	shifted, ctrld;
	static BYTE code;
	static BYTE shift_pending;

	// do not keyboard auto scan,
	//  during file select operation
	if(scan_inhibit) return;

	// previous key shift pending?
	if(shift_pending){
		shift_pending = 0;
		keymatrix[code / 8] = ~(0x01 << (code % 8));
		return;
	}

	// read USB keyboard
	if((key = keyboard_read()) == 0xffff){
		return; // do nothing, key not ready
	}

	// reset modifyer key flag
	shifted = ctrld = 0;

	// get key data(low byte)
	//  key: highbyte modifyer flag
	//       lowbyte  key data
	data = key&0xff;

	// key release or no input?
	if(!data){
		clear_keymatrix();
		return;
	}

	// Function Keys operation?
	if(data == FKEY11_PRESS){
		cpu_reset = 1;
		return;
	}else if(data == FKEY12_PRESS){
		cpu_nmi = 1;
		return;
	}else if(data >= FKEY1_PRESS && data <= FKEY8_PRESS){
		cpu_int = 1;
		cpu_intn = data - FKEY1_PRESS;
		return;
	}else if(data == FKEY10_PRESS){
		play_button = 0x10; // set BIT4 
		return;
	} 


	// check CTRL keys (only MZ-80A mode)
	if(machine_mode == MD_MZ80A && (key & (LEFT_CTRL_PRESS|RIGHT_CTRL_PRESS))){
		keymatrix[0]=0x7F; // ctrl
		ctrld = 1;
	}

	// check SHIFT keys
	if(key & LEFT_SHIFT_PRESS){
		if(machine_mode == MD_MZ80A) keymatrix[0]=0xFE;	// 'Shift'
		else keymatrix[8]=0xFE;	// 'Shift'
		shifted = 1;
	}else if(key & RIGHT_SHIFT_PRESS){
		if(machine_mode == MD_MZ80A) keymatrix[0]=0xFE;	// 'Shift'
		else keymatrix[8]=0xDE;	// 'Shift'
		shifted = 1;
	}

	// key data convert 
	if(data >= A_PRESS && data <= NZero_PRESS){
		code = code_convert[machine_mode][data - A_PRESS];
	}else if(data >= BACKSLASH_PRESS && data <= YEN_PRESS){
        code = code_convert[machine_mode][95 + (data - BACKSLASH_PRESS)];
	}else{
		return;  // unsupported key
	}

	// reconvert with Shift key
    if(shifted && (data >= EQUAL_PRESS && data <= QUEST_PRESS)){
		code = code_convert[machine_mode][100 + (data - EQUAL_PRESS)];
		if(!(code & 0x80)){ // need unshift,  at '='
			shifted = 0;
			keymatrix[0] = keymatrix[8] = 0xff;
		}
	}

	// unsupported key ?
	if(code == 255){
		return; // doesn't support
	}

	// need with shift ?
	if(!shifted && (code & 0x80)){ // need shift
		shifted=1;
		if(machine_mode == MD_MZ80A) keymatrix[0]=0xFE;	// set 'Shift'
		else keymatrix[8]=0xDE;	// set 'Shift'
	}
    code &= 0x7f; // clear shift flag

	// with ctrl ?
	if(ctrld){
		// delayed code setting.
		shift_pending = 1;
		return;
	}

	// not with shift ?
	if(!shifted){
		keymatrix[code / 8] = ~(0x01 << (code % 8));
		return;
	}

	// with shift
	if(machine_mode == MD_MZ80A && code/8 == 0){
		keymatrix[0] = ~(1|(0x01 << (code % 8)));
	}else if(machine_mode == MD_MZ1200 && code/8 == 8){
		keymatrix[8] = ~(0x21|(0x01 << (code % 8)));
	}else{
		// delayed code setting.
		shift_pending = 1;
	}
}


/*** end of key.c ****************************************************************************/
