/***************************************************************************************

	fselect.c

	mzf file select menu

	          1         2         3
	0123456789012345678901234567890123456789
	1
	2 
	3
	4     (7,5)                   (32,5) 
	5      +------------------------+
	6      | BC.MZF                 |
	7      | 8QUEEN.MZF             |
	8      | HUBASIC.MZF            |
	9      | LUNER LANDER.MZF       |
	10     | SPIDER MAZE.MZF        |
	11     | STAR TREK.MZF          |
	12     | HUCOMPILER.MZF         |
	13     | PASCAL.MZF             |
	14     | DISK-BASIC.MZF         |
	15     | Z80ASSEMBLER.MZF       |
	16     +------------------------+
	17    (7,16)                  (32,16)
	18

	2013/10/30  S.Suwa http://www.suwa-koubou.jp	

    2022/07/18  update menu reverse control
***************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "GenericTypeDefs.h"
#include "mz1200.h"
#include "key.h"
#include "mzfile.h"
#include "video.h"

static unsigned char scr_buf[1000];
static unsigned char rvs_buf[1000];   // 2022/07/18
void display_flist(int offset, int selected);

/*=====================================================================================
	code convert
=====================================================================================*/

// Ascii to Display code table
const unsigned char display_code[] = {
/* 00-0F */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* 10-1F */	0x00, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* 20-2F */	0x00, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6B, 0x6A, 0x2F, 0x2A, 0x2E, 0x2D,
/* 30-3F */	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x4F, 0x2C, 0x51, 0x2B, 0x57, 0x49,
/* 40-4F */	0x55, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
/* 50-5F */	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x52, 0x59, 0x54, 0x50, 0x45,
/* 60-6F */	0xC7, 0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 0xDF, 0xE7, 0xE8, 0xE9, 0xEA, 0xEC, 0xED,
/* 70-7F */	0xD0, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xC0,
/* 80-8F */	0x00, 0xBD, 0x9D, 0xB1, 0xB5, 0xB9, 0xB4, 0x9E, 0xB2, 0xB6, 0xBA, 0xBE, 0x9F, 0xB3, 0xB7, 0xBB,
/* 90-9F */	0xBF, 0xA3, 0x85, 0xA4, 0xA5, 0xA6, 0x94, 0x87, 0x88, 0x9C, 0x82, 0x98, 0x84, 0x92, 0x90, 0x83,
/* A0-AF */	0x91, 0x81, 0x9A, 0x97, 0x93, 0x95, 0x89, 0xA1, 0xAF, 0x8B, 0x86, 0x96, 0xA2, 0xAB, 0xAA, 0x8A,
/* B0-BF */	0x8E, 0xB0, 0xAD, 0x8D, 0xA7, 0xA8, 0xA9, 0x8F, 0x8C, 0xAE, 0xAC, 0x9B, 0xA0, 0x99, 0xBC, 0xB8,
/* C0-CF */	0x00, 0x3B, 0x3A, 0x70, 0x3C, 0x71, 0x5A, 0x3D, 0x43, 0x56, 0x3F, 0x1E, 0x4A, 0x1C, 0x5D, 0x3E,
/* D0-DF */	0x5C, 0x1F, 0x5F, 0x5E, 0x37, 0x7B, 0x7F, 0x36, 0x7A, 0x7E, 0x33, 0x4B, 0x4C, 0x1D, 0x6C, 0x5B,
/* E0-EF */	0x78, 0x41, 0x35, 0x34, 0x74, 0x30, 0x38, 0x75, 0x39, 0x4D, 0x6F, 0x6E, 0x32, 0x77, 0x76, 0x72,
/* F0-FF */	0x73, 0x47, 0x7C, 0x53, 0x31, 0x4E, 0x6D, 0x48, 0x46, 0x7D, 0x44, 0x1B, 0x58, 0x79, 0x42, 0x60,
};


// small alphabet code convert to MZ-80A Ascii code
const unsigned char toMZAscii[] = {
//   a    b    c    d    e    f    g    h
  0xa1,0x9a,0x9f,0x9c,0x92,0xaa,0x97,0x98,
//   i    j    k    l    m    n    o    p
  0xa6,0xaf,0xa9,0xb8,0xb3,0xb0,0xb7,0x9e,
//   q    r    s    t    u    v    w    x    y    z
  0xa0,0x9d,0xa4,0x96,0xa5,0xab,0xa3,0x9b,0xbd,0xa2,
};


// from  standard ascii code to MZ ascii code 
unsigned char asc2mzasc(unsigned char c)
{
	if(machine_mode == MD_MZ80A){
		if(c <= 0x5d) return c; // numeric, alphabet
		if(c >= 'a' && c <= 'z') return toMZAscii[c - 'a']; // small alphabet
		if(c == '^') return 0x8B;
		if(c == '{') return 0xBE;
		if(c == '}') return 0x80;
		if(c == '|') return 0xC0;
		if(c == '~') return 0x94;
		if(c == '`') return 0x93;
		return 0x20; // space
	} 

	if(machine_mode == MD_MZ1200){
		if(c <= 0x5d) return c; // numeric, alphabet
		if(c >= 'a' && c <= 'z') return c - 0x20; // to upper
		if(c >= 0xA0 && c <= 0xDF) return c - 0x20; // KANA 
		if(c == '_') return 0xc4; 
		if(c == '|') return 0xe2;
		return 0x20; // space
	}
}


/*=====================================================================================
	file select

  return code = 0   complete
              = -1  cancel, (break key press)
=====================================================================================*/

static int prev_select = -1;  // 2022/07

int file_select(void)
{
	unsigned char  c;
	int i, p, row, col, rc;

	WORD key, prev;
	static int	selected, offset;


	// lock keyboard auto scan
	scan_inhibit = 1;
	delay_ms(20);	// wait next keyboard auto scan

	// save now screen
	for(i = 0; i < 1000; i++){
		scr_buf[i] = RAM[(VMem+i)&0xD7FF];
        rvs_buf[i] = Vrvs[i]; // 2022/07
	}


	// display list frame
	p = VMem+(5*40);
	for(row = 5; row <= 16; row++, p+=40){
		for(col = 7; col <= 32; col++){
			if(row == 5 && col == 7){         // upper left corner
				c = 0x5c;
			}else if(row == 5 && col == 32){  // upper right corner
				c = 0x5d;
			}else if(row == 16 && col == 7){  // lower left corner
				c = 0x1c;
			}else if(row == 16 && col == 32){ // lower right corner
				c = 0x1d;
			}else if(row == 5 || row == 16){  // horizontal line
				c = 0x78;
			}else if(col == 7 || col == 32){  // vartical line
				c = 0x79;
			}else{
				c = 0x00;                     // fill space in list area
			}
			RAM[(p+col)&0xD7FF] = c;  // write display code
		}
	}


	prev = 0xffff;
    prev_select = -1;  // 2022/07

	for(;;){
		// display file list
		display_flist(offset, selected);

		// wait key input
		while((key = keyboard_read()) == prev) delay_ms(10);
		prev = key;

		switch(key & 0xff){
		case UP_PRESS:
			if(!selected && offset) offset--;
			if(selected) selected--;
			break;	

		case DOWN_PRESS:
			if(selected < 9) selected++;
			else if(offset < inf_max-1) offset++;
			break;

		case ENTER_PRESS:
			if(offset+selected < inf_max) goto exit_for;
			break;

		case BREAK_PRESS:
		case ESCAPE_PRESS:
			rc = -1;
			play_button = 0; // stop motor
			goto can_exit;
		} // switch

	} //for

exit_for:
	rc = 0;
	inf_next = offset+selected;
can_exit:

	// reload saved screen
	for(i = 0; i < 1000; i++){
		RAM[(VMem+i)&0xD7FF] = scr_buf[i];
        Vrvs[i] = rvs_buf[i];  // 2022/07
	}

	// unlock keyboard auto scan
	scan_inhibit = 0;
	return rc;
}


/*--------------------------------------------------------------------------------------
	display file list
--------------------------------------------------------------------------------------*/

void display_flist(int offset, int selected)
{
	int line, i, n, len, p;
	unsigned char *fn;
	unsigned char c;

	// display list 
	for(line=0, n=offset; n < inf_max && line < 10; n++, line++){

		p = VMem+(6+line)*40+9;

		// write space
		for(i = 0; i < 22; i++)	RAM[(p+i)&0xD7FF]=0x00;

		// write file name, small to upper convert
		fn = inf_file[n];
		len = strlen(fn);
		for(i = 0; i < len && i < 22; i++, fn++){
			c = display_code[asc2mzasc(*fn)];
			RAM[(p+i)&0xD7FF]=c; // write display code
		}
	}


	// blank line
	for( ;line < 10; line++){
		p = VMem+(6+line)*40+9;
		// write space
		for(i = 0; i < 22; i++)	RAM[(p+i)&0xD7FF]=0x00;
	}

	// video signal control
	//  selected file is displayed reverse
    //  update 2022/07/18
    if(prev_select != selected){
        if(prev_select != -1){
            for(i=0; i<24; i++){
                Vrvs[(prev_select +6)*40+8+i]=scrnRvrs ? 0xff: 00;
            }
        }
        for(i=0; i<24; i++){
            Vrvs[(selected +6)*40+8+i]=scrnRvrs ? 0x00: 0xff;
        }
        prev_select = selected;
    }
}


/*** end of fselect.c *****************************************************************/


