/******************************************************************************
 
  USB Host UVC Driver

	S.Suwa       31-Jul-2012 First release
                 18-Jan-2013 add MJPEG format etc
                  6-Apl-2013 modified Transfer Complete Check Routine,
                             and Iscochronous Transfer Routine
	             http://www.suwa-koubou.jp

*******************************************************************************/

#include <stdlib.h>
#include <string.h>
#include "GenericTypeDefs.h"
#include "usb.h"
#include "usb_host_uvc.h"

// print out for DEBUG
//#define DEBUG_MODE
#ifdef DEBUG_MODE
    #include "uart2.h"
	#define	putstr(x)	UART2PrintString(x)
	#define	inkey()		UART2IsPressed()
	#define	getch()		UART2GetChar()
	#define	puthex(x)	UART2PutHex(x)
	#define	putch(x)	UART2PutChar(x)
	#define putcrlf()   UART2PrintString("\r\n")
#endif

/*------------------------------------------------------------------------------
	macro Constants
-------------------------------------------------------------------------------*/

// Video Interface Class Code
#define	CC_VIDEO				0x0e

// Video Interface Sub Class Codes
#define	SC_UNDEFINED			0x00
#define	SC_VIDEOCONTROL			0x01
#define	SC_VIDEOSTREAMING		0x02
#define	SC_VIDEO_INTERFACE_COLLECTION 0x03

// Video Interface Protocol Codes
#define	PC_PROTOCOL_UNDEFINED	0x00

// Video Class-Specific Descriptor Type
#define	CS_UNDEFINED		0x20
#define CS_DEVICE			0x21
#define CS_CONFIGURATION	0x22
#define CS_STRING			0x23
#define CS_INTERFACE		0x24
#define CS_ENDPOINT			0x25

// Video Class-Specific VC Interface Descriptor Subtypes
#define VC_DESCRIPTOR_UNDEFINED 0x00
#define	VC_HEADER				0x01
#define VC_INPUT_TERMINAL		0x02
#define VC_OUTPUT_TERMINAL		0x03
#define VC_SELECTOR_UNIT		0x04
#define VC_PROCESSING_UNIT		0x05
#define VC_EXTENSION_UNIT		0x06

// Video Class-Specific VS Interface Descriptor Subtypes
#define VS_UNDEFINED			0x00
#define VS_INPUT_HEADER			0x01
#define VS_OUTPUT_HEADER		0x02
#define VS_STILL_IMAGE_FRAME	0x03
#define VS_FORMAT_UNCOMPRESSED	0x04
#define VS_FRAME_UNCOMPRESSED	0x05
#define VS_FORMAT_MJPEG			0x06
#define VS_FRAME_MJPEG			0x07
#define VS_FORMAT_MPEG2TS		0x0a
#define VS_FORMAT_DV			0x0c
#define VS_COLORFORMAT			0x0d
#define VS_FORMAT_FRAME_BASED	0x10
#define VS_FRAME_FRAME_BASED	0x11
#define VS_FORMAT_STREAM_BASED	0x12

// Video Class-Specific Endpoint Descriptor Subtypes
#define	EP_UNDEFINED			0x00
#define	EP_GENERAL				0x01
#define	EP_ENDPOINT				0x02
#define	EP_INTERRUPT			0x03

// Terminal Type
// *** USB Terminal Types
#define	TT_VENDOR_SPECIFIC		0x0100
#define	TT_STREAMING			0x0101
// *** Input Terminal Types
#define	ITT_VENDOR_SPECIFIC		0x0200
#define ITT_CAMERA				0x0201
#define	ITT_MEDIA_TRANSPORT_INPUT 0x0202
// *** OutPut Terminal Types
#define	OTT_VENDOR_SPECIFIC		0x0300
#define	OTT_DISPLAY				0x0301
#define	OTT_MEDIA_TRANSPORT_OUTPUT 0x0302
// *** External Terminal Types
#define	EXTERNAL_VENDOR_SPECIFIC 0x0400
#define	COMPOSITE_CONNECTOR		0x0401
#define	SVIDEO_CONNECTOR		0x0402
#define	COMPONENT_CONNECTOR		0x0403

// UVC Control Selector Codes
// *** VideoControl Interface Control Selectors
#define	VC_CONTROL_UNDEFINED	0x00
#define	VC_VIDEO_POWER_MODE		0x01
#define	VC_REQUEST_ERROR		0x02
// *** Terminal Control Selectors
#define	TE_CONTROL_UNDEFINED	0x00
// *** Selector Unit Control Slectors
#define	SU_CONTROL_UNDEFINED	0x00
#define	SU_INPUT_SELECT			0x01
// *** Camera Terminal Control Selectors
#define CT_CONTROL_UNDEFINED	0x00
#define CT_SCANNING_MODE		0x01
#define CT_AE_MODE				0x02
#define CT_AE_PRIORITY			0x03
#define CT_EXPOSURE_TIME_ABSOLUTE 0x04
#define CT_EXPOSURE_TIME_RELATIVE 0x05
#define CT_FOCUS_ABSOLUTE		0x06
#define CT_FOCUS_RELATIVE		0x07
#define CT_FOCUS_AUTO			0x08
#define CT_IRIS_ABSOLUTE		0x09
#define CT_IRIS_RELATIVE		0x0a
#define CT_ZOOM_ABSOLUTE		0x0b
#define CT_ZOOM_RELATIVE		0x0c
#define CT_PANTILT_ABSOLUTE		0x0d
#define CT_PANTILT_RELATIVE		0x0e
#define CT_ROLL_ABSOLUTE		0x0f
#define CT_ROLL_RELATIVE		0x10
#define CT_PRIVACY				0x11
// *** Processing Unit Control Selectors
#define	PU_CONTROL_UNDEFINED	0x00
#define	PU_BACKLIGHT_COMPENSATION 0x01
#define	PU_BRIGHTNESS			0x02
#define	PU_CONTRAST				0x03
#define	PU_GAIN					0x04
#define	PU_POWER_LINE_FREQUENCY	0x05
#define	PU_HUE					0x06
#define	PU_SATURATION			0x07
#define	PU_SHARPNESS			0x08
#define	PU_GAMMA				0x09
#define	PU_WHITE_BALANCE_TEMPERATURE 0x0a
#define	PU_WHITE_BALANCE_TEMPERATURE_AUTO 0x0b
#define	PU_WHITE_BALANCE_COMPONENT 0x0c
#define	PU_WHITE_BALANCE_COMPONENT_AUTO 0x0d
#define	PU_MULTIPLIER			0x0e
#define	PU_MULTIPLIER_LIMIT		0x0f
#define	PU_HUE_AUTO				0x10
#define	PU_ANALOG_VIDEO_STANDARD 0x11
#define	PU_ANALOG_LOCK_STATUS	0x12
// *** Extension Unit Control Selectors
#define	XU_CONTROL_UNDEFINED	0x00
// *** VideoStreaming Interface Control Selectors
#define	VS_CONTROL_UNDEFINED	0x00
#define VS_PROBE				0x01
#define VS_COMMIT				0x02
#define VS_STILL_PROBE			0x03
#define VS_STILL_COMMIT			0x04
#define VS_STILL_IMAGE_TRIGGER	0x05
#define VS_STREAME_ERROR_CODE	0x06
#define VS_GENERATE_KEY_FRAME	0x07
#define VS_UPDATE_FRAME_SEGMENT	0x08
#define VS_SYNCH_DELAY			0x09

// image format guid
#define	GUID_FMT_YUY2	"\x59\x55\x59\x32\x00\x00\x10\x00\x80\x00\x00\xAA\x00\x38\x9B\x71"
#define	GUID_FMT_NV12	"\x4e\x56\x31\x32\x00\x00\x10\x00\x80\x00\x00\xAA\x00\x38\x9B\x71"


// Video Class-Specific Request Codes
// below definition include usb_host_uvc.h
//#define	RC_UNDEFIND	0
//#define	SET_CUR		1
//#define	GET_CUR		0x81
//#define	GET_MIN		0x82
//#define	GET_MAX		0x83
//#define	GET_RES		0x84
//#define	GET_LEN		0x85
//#define	GET_INFO 	0x86
//#define	GET_DEF		0x87

// DataType Codes 
// below definition include usb_host_uvc.h
//#define	BOOLEAN		1
//#define	bBITMAP		2
//#define	wBITMAP		3
//#define	bSIGNED		4
//#define	wSIGNED		5
//#define	dwSIGNED	6
//#define	bNUMBER		7
//#define	wNUMBER		8
//#define	dwNUMBER	9


/*------------------------------------------------------------------------------
     Type Definition
-------------------------------------------------------------------------------*/

typedef struct {
	char	*name;		// Control display name
	BYTE	bRequest;	// bitmap valid request
		/*
		 * bRequest: bitmap mandatory request code
		 *  D0 SET_CUR
		 *  D1 GET_CUR
		 *  D2 GET_MIN
		 *  D3 GET_MAX
		 *  D4 GET_RES
		 *  D5 GET_LEN
		 *  D6 GET_INFO
		 *  D7 GET_DEF
		 */
	WORD	wValue;		// control code in UVC Class-Specific Codes
	WORD	wIndex;		// UnitID, TerminalID, Interface Number in descriptor
	WORD	wLength;	// length of data
	WORD	wDataType;	// data type
		/* wDataType  halfbyte x 4
		 * ex. length of data is 4byte
		 *	   (bNUMBER << 12)|(bSIGNED << 8)|(bNUMBER <<4)|bSIGNED
		 *     length of data is 8byte
		 *     (dwSIGNED << 4)|dwSIGNED
		 */
} UVC_REQUEST;


typedef struct	{
	WORD	bmHint;
	BYTE	bFormatIndex;
	BYTE	bFrameIndex;
	DWORD	dwFrameInterval;
	WORD	wKeyFrameRate;
	WORD	wPFrameRate;
	WORD	wCompQuality;
	WORD	wCompWindowSize;
	WORD	wDelay;
	DWORD	dwMaxVideoFrameSize;
	DWORD	dwMaxPayloadTransferSize;
	DWORD	dwClockFrequency;
	BYTE	bmFramingInfo;
	BYTE	bPreferedVersion;
	BYTE	bMinVersion;
	BYTE	bMaxVersion;
}	
#if defined(__PIC32MX__)
 __attribute__ ((packed)) VS_PROBE_DATA; 
#else
 VS_PROBE_DATA;	
#endif


/*------------------------------------------------------------------------------
    Global Variables
-------------------------------------------------------------------------------*/

const UVC_REQUEST UVC_Control[] = {
	// VideoControl Interface Control Requests 
	{ "Video Power",            0b01000011, VC_VIDEO_POWER_MODE << 8,      0,  1, bBITMAP, },
	{ "Request Error Code",     0b01000010, VC_REQUEST_ERROR << 8,         0,  1, bNUMBER, },
	// Camera Terminal Control Requests
	{ "Scanning Mode",          0b01000011, CT_SCANNING_MODE << 8,          VC_INPUT_TERMINAL, 1, BOOLEAN, }, 
	{ "Auto-Exposure Mode",     0b11010011, CT_AE_MODE << 8,                VC_INPUT_TERMINAL, 1, bBITMAP, },
	{ "Auto-Exposure Priority", 0b01000011, CT_AE_PRIORITY << 8,            VC_INPUT_TERMINAL, 1, bNUMBER, },
	{ "Exposure Time(Absolute)",0b11011110, CT_EXPOSURE_TIME_ABSOLUTE << 8, VC_INPUT_TERMINAL, 4, dwNUMBER, },
	{ "Exposure Time(Relative)",0b01000011, CT_EXPOSURE_TIME_RELATIVE << 8, VC_INPUT_TERMINAL, 1, bSIGNED, },
	{ "Focus(Absolute)",        0b11011110, CT_FOCUS_ABSOLUTE << 8,         VC_INPUT_TERMINAL, 2, wNUMBER, },
	{ "Focus(Relative)",        0b11011111, CT_FOCUS_RELATIVE << 8,         VC_INPUT_TERMINAL, 1, bSIGNED, },
	{ "Iris(Absolute)",         0b11011110, CT_IRIS_ABSOLUTE << 8,          VC_INPUT_TERMINAL, 2, wNUMBER, },
	{ "Iris(Relative)",         0b01000011, CT_IRIS_RELATIVE << 8,          VC_INPUT_TERMINAL, 1, bSIGNED, },
	{ "Zoom(Absolute)",         0b11011110, CT_ZOOM_ABSOLUTE << 8,          VC_INPUT_TERMINAL, 2, wNUMBER},
	{ "Zoom(Relative)",         0b11011111, CT_ZOOM_RELATIVE << 8,          VC_INPUT_TERMINAL, 3, (bNUMBER << 8)|(BOOLEAN << 4)|bSIGNED, },
	{ "Pantilt(Absolute)",      0b11011110, CT_PANTILT_ABSOLUTE << 8,       VC_INPUT_TERMINAL, 8, (dwSIGNED << 4)|dwSIGNED, },
	{ "Pantilt(Relative)",      0b11011111, CT_PANTILT_RELATIVE << 8,       VC_INPUT_TERMINAL, 4, (bNUMBER << 12)|(bSIGNED << 8)|(bNUMBER <<4)|bSIGNED, },
	{ "Roll(Absolute)",         0b11011110, CT_ROLL_ABSOLUTE << 8,          VC_INPUT_TERMINAL, 2, wSIGNED, },
	{ "Roll(Relative)",         0b11011111, CT_ROLL_RELATIVE << 8,          VC_INPUT_TERMINAL, 2, (bNUMBER << 4)|bSIGNED, },
	{ "Reserved",               0,          0,                              0,                 0, 0, },
	{ "Reserved",               0,          0,                              0,                 0, 0, },
	{ "Focus,Auto",             0b11000011, CT_FOCUS_AUTO << 8,             VC_INPUT_TERMINAL, 1, BOOLEAN, },
	{ "Privacy",                0b01000010, CT_PRIVACY << 8,                VC_INPUT_TERMINAL, 1, BOOLEAN, },
	// Selector Unit Control Requests
	{ "Input Select",           0b01011111, SU_INPUT_SELECT << 8,	        VC_SELECTOR_UNIT,  1, bNUMBER, },		  
	// Processing Unit Control Requests
	{ "Brightness",                     0b11011111, PU_BRIGHTNESS << 8,             VC_PROCESSING_UNIT, 2, wSIGNED, },
	{ "Contrast",                       0b11011111, PU_CONTRAST << 8,               VC_PROCESSING_UNIT, 2, bNUMBER, },
	{ "Hue",                            0b11011110, PU_HUE << 8,                    VC_PROCESSING_UNIT, 2, wSIGNED, },
	{ "Saturation",                     0b11011111, PU_SATURATION << 8,             VC_PROCESSING_UNIT, 2, wNUMBER, },
	{ "Sharpness",                      0b11011111, PU_SHARPNESS << 8,              VC_PROCESSING_UNIT, 2, wNUMBER, },
	{ "Gamma",                          0b11011111, PU_GAMMA << 8,                  VC_PROCESSING_UNIT, 2, wNUMBER, },
	{ "White Balance Temperature",      0b11011110, PU_WHITE_BALANCE_TEMPERATURE << 8, VC_PROCESSING_UNIT, 2, wNUMBER, },
	{ "White Balance Component",        0b11011110, PU_WHITE_BALANCE_COMPONENT << 8, VC_PROCESSING_UNIT, 4, (wNUMBER << 4)|wNUMBER, },
	{ "Backlight Compensation",         0b11011111, PU_BACKLIGHT_COMPENSATION << 8, VC_PROCESSING_UNIT, 2, wNUMBER, },
	{ "Gain",                           0b11011111, PU_GAIN << 8,                   VC_PROCESSING_UNIT, 2, wNUMBER, },
	{ "Power Line Frequency",           0b11000011, PU_POWER_LINE_FREQUENCY << 8,   VC_PROCESSING_UNIT, 1, bBITMAP, },
	{ "Hue,Auto",                       0b11000011, PU_HUE_AUTO << 8,               VC_PROCESSING_UNIT, 1, bNUMBER, },
	{ "White Balance Temperature,Auto", 0b11000011, PU_WHITE_BALANCE_TEMPERATURE_AUTO << 8, VC_PROCESSING_UNIT, 1, bNUMBER,  },
	{ "White Balance Component,Auto",   0b11000011, PU_WHITE_BALANCE_COMPONENT_AUTO << 8, VC_PROCESSING_UNIT, 1, bNUMBER, },
	{ "Digital Multiplier",             0b11011111, PU_MULTIPLIER << 8,             VC_PROCESSING_UNIT, 2, wNUMBER, },
	{ "Digital Multiplier Limit",       0b11011111, PU_MULTIPLIER_LIMIT << 8,       VC_PROCESSING_UNIT, 2, wNUMBER, },
	{ "Analog Video Standard",          0b01000010, PU_ANALOG_VIDEO_STANDARD << 8,  VC_PROCESSING_UNIT, 1, bNUMBER, },
	{ "Analog Video Lock Status",       0b01000010, PU_ANALOG_LOCK_STATUS << 8,     VC_PROCESSING_UNIT, 1, bNUMBER, },
	// Video Streaming Interface Control Requests
	{ "Video Probe",                    0b11111111, VS_PROBE << 8,                   1, 34, 0, },
	{ "Video Commit",                   0b01100011, VS_COMMIT << 8,                  1, 34, 0, },
	{ "Video Still Probe",              0b11101111, VS_STILL_PROBE << 8,             1, 11, 0, },
	{ "Video Still Commit",             0b01100011, VS_STILL_COMMIT << 8,            1, 11, 0, },
	{ "Video Still Trigger",            0b01000011, VS_STILL_IMAGE_TRIGGER << 8,     1, 1, bNUMBER, },
	{ "Video Error Code",               0b01000010, VS_STREAME_ERROR_CODE << 8,      1, 1, bNUMBER, },
	{ "Video Generate Key Frame",       0b01000011, VS_GENERATE_KEY_FRAME << 8,      1, 1, bNUMBER, },
	{ "Video Update Frame",             0b11011111, VS_UPDATE_FRAME_SEGMENT << 8,    1, 2, (bNUMBER <<4)|bNUMBER, },
	{ "Video Synch Delay",              0b11011111, VS_SYNCH_DELAY << 8,             1, 2, wNUMBER, },
};

/*
 * Unit or Terminal ID and Interface Number table
 *   wUT_ID[0] is VideoControl Interface Number.
 *   wUT_ID[1] is VideoStreaming Interface Number.
 *   wUT_ID[VC_INPUT_TERMINAL(2)]  is Camera Terminal ID.
 *   wUT_ID[VC_OUTPUT_TERMINAL(3)] is OUTPUT Terminal ID.
 *   wUT_ID[VC_SELECTOR_UNIT(4)]   is Selector Unit ID.
 *   wUT_ID[VC_PROCESSING_UNIT(5)] is Camera Processing Unit ID.
 *   wUT_ID[VC_EXTENSION_UNIT(6)]  is EXTENSION Unit ID.
 */
static WORD wUT_ID[7];

static BYTE InterfaceNum[2]; // InterfaceNum[0]=VideoControl Interface Number
                             // InterfaceNum[1]=VideoStreaming Interface Number
static DWORD CT_bmControls; // Valid D0-D18  19bits
static DWORD PU_bmControls; // Valid D0-D17  18bits
static BYTE	UVC_EP[2];      // UVC_EP[0] = VideoControlInterface Interrupt IN Endpoint Number
                            // UVC_EP[1] = VideoStreaming Isochronous IN Endpoint Number	

static WORD UVC_version;	// UVC version of device, bcdUVC value is 0x0110 or 0x0100
static BYTE CaptureMethod;	// StillCaptureMethod is Method1(0x01)/Method2(0x02)/Method3(0x03).
static BYTE NumFormats;     // numbers of format

static BYTE NumAlternates;				// count of VideoStreaming Alternate Interfaces
static WORD *AlternateMaxPacketSize;	// array of Endpoint MaxpacketSize in Alternate Interface
			// array[0] is zero, valid data is from array[1] to array[NumAlternates]

static volatile int TransferError_FLG;		// flag of Transfer error

static WORD VendorId;               // Vendor ID of the device
static WORD ProductId;              // Product ID of the device
static BYTE deviceAddress;          // Address of the device on the USB
static BYTE	driverId;               // this Driver Id
static BYTE	driverInitialized;		// Driver has been initialized

static	UVC_FORMAT_INFO	*VSFormat;

#ifdef DEBUG_MODE
	static	char	*format_name[] = {
		"YUYV", "NV12", "MJPEG", "MPEG2 TS", "DV",
	};
#endif


/*------------------------------------------------------------------------------
    internal Function Prototype
-------------------------------------------------------------------------------*/

static	BYTE	parse_configuration_descriptor(void);

/*=============================================================================
    BOOL USBHostUVCInit ( BYTE address, DWORD flags, BYTE clientDriverID )

  Parameters:
    BYTE address    - Device's address on the bus
    DWORD flags     - Initialization flags
    BYTE clientDriverID - ID to send when issuing a Device Request via
                       USBHostIssueDeviceRequest(), USBHostSetDeviceConfiguration(),
                       or USBHostSetDeviceInterface().  

  Return Values:
    TRUE    - Initialization was successful
    FALSE   - Initialization failed

==============================================================================*/

BOOL USBHostUVCInit ( BYTE address, DWORD flags, BYTE clientDriverID )
{
    BYTE	*pDesc, n;

	if(driverInitialized) return TRUE;

    // Save device the address, VendorId, & ProductId
    deviceAddress = address;
    pDesc  = USBHostGetDeviceDescriptor(address);
    pDesc += 8;
    VendorId  =  (WORD)*pDesc;       pDesc++;
    VendorId |= ((WORD)*pDesc) << 8; pDesc++;
    ProductId =  (WORD)*pDesc;       pDesc++;
    ProductId |= ((WORD)*pDesc) << 8; pDesc++;

    // Save the Client Driver ID
    driverId = clientDriverID;

	// reset parameter to clear 
	for(n = 0; n < sizeof(wUT_ID)/sizeof(wUT_ID[0]); n++) wUT_ID[n] = 0;
	InterfaceNum[0] = InterfaceNum[1] = 0;
	CT_bmControls = PU_bmControls = 0L;
	UVC_EP[0] = UVC_EP[1] = 0;

	// free memory 
	if(VSFormat != NULL){
		if(NumFormats == 0) return FALSE;
		for(n = 0; n < NumFormats; n++){
			if(VSFormat[n].frame) free(VSFormat[n].frame);
		}
		free(VSFormat);
		VSFormat = NULL;
	}
	NumFormats = 0;

	if(AlternateMaxPacketSize != NULL) free(AlternateMaxPacketSize);
	AlternateMaxPacketSize = NULL;
	NumAlternates = 0;


	// Parse Configuration and Get parameter
	if(parse_configuration_descriptor() == UVC_FAIL){
		return FALSE;
	}

    // UVC Client Driver Init Complete.
    driverInitialized = 1;

    #ifdef DEBUG_MODE
	{
		static char buf[128];

		BYTE *pConDesc;
		WORD length, n, m, width, height;
		void	dump_descriptor(unsigned char *p, int len);

        putstr( "UVC: USB UVC Client Initalized.\r\n");
		sprintf(buf, "     flags=0x%04x, address=%d, VID=0x%04x, PID=0x%04x\r\n",
                           (WORD)flags, deviceAddress, VendorId, ProductId);
		putstr(buf);		

		putstr("===== Device Descriptor ==============================\r\n");
	    pDesc  = USBHostGetDeviceDescriptor(deviceAddress);
		length = pDesc[0];	// bLength
		dump_descriptor(pDesc, pDesc[0]);
		putstr("===== Configuration Descriptor =======================\r\n");
		pConDesc = USBHostGetCurrentConfigurationDescriptor(deviceAddress);
		length = (pConDesc[3] << 8) | pConDesc[2];
		dump_descriptor(pConDesc, length); 
		putstr("======================================================\r\n");

		sprintf(buf, "InterfaceNum[0]=0x%02x\r\n", InterfaceNum[0]); putstr(buf);
		sprintf(buf, "InterfaceNum[1]=0x%02x\r\n", InterfaceNum[1]); putstr(buf);
		sprintf(buf, "Endpoint[0]=0x%02x\r\n", UVC_EP[0]); putstr(buf);
		sprintf(buf, "Endpoint[1]=0x%02x\r\n", UVC_EP[1]); putstr(buf);
		for(n = 0; n < sizeof(wUT_ID)/2; n++){
			 sprintf(buf, "wUT_ID[%d]=0x%04x\r\n", n, wUT_ID[n]); putstr(buf);
		}
		sprintf(buf, "CT_bmControls=0x%06lx\r\n", CT_bmControls); putstr(buf);
		sprintf(buf, "PU_bmControls=0x%06lx\r\n", PU_bmControls); putstr(buf);

		for(n = 0; n < NumFormats; n++){
			sprintf(buf, "FormatIndex(%d) Format: ", VSFormat[n].bFormatIndex);
			putstr(buf);
			if(VSFormat[n].format <= FMT_DV){
				putstr(format_name[VSFormat[n].format]); putcrlf();
			}else{
				putstr("unknown\r\n");
			}

			for(m = 0; m < VSFormat[n].bNumFrames; m++){
				width = VSFormat[n].frame[m].wWidth;
				height =VSFormat[n].frame[m].wHeight;
				sprintf(buf, "\tFrameIndex(%d) Size=%dx%d", VSFormat[n].frame[m].bFrameIndex, width, height); putstr(buf);
				sprintf(buf, "\tInterval=0x%08lx,%dms(%dfps)\r\n", VSFormat[n].frame[m].dwFrameInterval,
					(int)(VSFormat[n].frame[m].dwFrameInterval/10000L), (int)(1000/(VSFormat[n].frame[m].dwFrameInterval/10000)) ); putstr(buf);
			}
		}

		for(n = 1; n < NumAlternates+1; n++){
			sprintf(buf, "EP_MaxPacketSize=%d\r\n", AlternateMaxPacketSize[n]); putstr(buf);
		}		

	}
	#endif


    // Notify that application that we've been attached to a device.
    USB_HOST_APP_EVENT_HANDLER(address, EVENT_UVC_ATTACH, NULL, 0 );


    return TRUE;

} // USBHostUVCInit


/*==============================================================================================
	Device infromation
	USBHostUVCDeviceAttached(void)
==============================================================================================*/

BYTE USBHostUVCDeviceAttached(void)
{
	if(deviceAddress && driverInitialized){
		return USB_DEVICE_ATTACHED;
	}else{
		return USB_DEVICE_DETACHED;
	}
}


/*==============================================================================================
	Get Image Format
	returned FMT_YUY2/FMT_NV12/FMT_MJPEG/FMT_DV/ etc.
==============================================================================================*/

BYTE	USBHostUVCGetFormat(int number)
{
	return VSFormat[number].format;
}


/*==============================================================================================
	Get Image Format counts
	return number of formats
==============================================================================================*/

BYTE	USBHostUVCGetFormatCount(void)
{
	return NumFormats;
}


/*==============================================================================================
	Get Image resolution
	returned array of [Width x Height] 
==============================================================================================*/

UVC_FRAME_INFO *USBHostUVCGetFrame(BYTE format_number, BYTE *numframes)
{
	if(format_number >= NumFormats) return NULL;

	*numframes = VSFormat[format_number].bNumFrames;
	return VSFormat[format_number].frame;
}



/*==============================================================================================
	get Control Name
	control value is from UVC_APL_POWER_MODE to UVC_APL_SYNCH_DELAY.
==============================================================================================*/

char *USBHostUVCGetControlName(BYTE control)
{
	// Check control range
	if(control > UVC_APL_SYNCH_DELAY) return NULL;

	// return with name string pointer
	return UVC_Control[control].name;
}


/*==============================================================================================
	get Control Value

	control: UVC_APL_POWER_MODE -- UVC_APL_SYNCH_DELAY
	request: GET_CUR/GET_MIN/GET_MAX/GET_DEF/GET_INFO
	data:    request result data
	len:	 length of valid data in buffer 
==============================================================================================*/

BYTE	USBHostUVCGetControlValue(BYTE control, BYTE request, BYTE *data, BYTE *len, WORD *type)
{
	BYTE	errorCode, size;
    DWORD   byteCount;

	// Check control range
	if(control > UVC_APL_SYNCH_DELAY)
		return UVC_INVALID;

	// check request code
	if(request < GET_CUR || request > GET_DEF) return UVC_INVALID;

	// check support in unit or terminal
	if(control > UVC_APL_ERROR_CODE && control < UVC_APL_INPUT_SELECT){
		// CT
		if((CT_bmControls & (1 << (control -UVC_APL_SCANNING))) == 0){
			// doesn't support control
			return UVC_UNSUPPORTED_CONTROL;
		}
	}else if(control > UVC_APL_INPUT_SELECT && control < UVC_APL_PROBE){
		// PU
		if((PU_bmControls & (1 << (control -UVC_APL_BRIGHTNESS))) == 0){
			// doesn't support control
			return UVC_UNSUPPORTED_CONTROL;
		}
	}

	// check support request
	if( (UVC_Control[control].bRequest & (1 << (request - 0x80))) == 0){
		return UVC_UNSUPPORTED_REQUEST;
	}


	// fixup data length
	if(control == UVC_APL_PROBE || control == UVC_APL_COMMIT){
		size = UVC_version >= 0x0110 ? 34: 26;
	}else{
		size = UVC_Control[control].wLength;
	}
	if(request == GET_INFO || request == GET_LEN){
		size = 1;
	}
	*len = size;

	// get data from device
	*type = UVC_Control[control].wDataType;
	if((errorCode = USBHostIssueDeviceRequest(
		deviceAddress,                       // address
		USB_SETUP_DEVICE_TO_HOST|USB_SETUP_TYPE_CLASS|USB_SETUP_RECIPIENT_INTERFACE, // bmRequestType
		request,					         // bRequest
		UVC_Control[control].wValue,         // wValue
		wUT_ID[UVC_Control[control].wIndex], // wIndex
		(WORD)size,                          // wLength
		data,                                // data
		USB_DEVICE_REQUEST_GET,              // direction
		driverId                             // clientDriverID
		           )) == USB_SUCCESS){
        while(USBHostTransferIsComplete( deviceAddress, 0, &errorCode, &byteCount ) != TRUE){
            USBHostTasks();
        }
    }

	if(errorCode != USB_SUCCESS){
		USBHostClearEndpointErrors( deviceAddress, 0 );
		#ifdef DEBUG_MODE
		    putstr("Control Get request error!\r\n");
		#endif
		return UVC_FAIL;
	}

	// request success
	return UVC_SUCCESS;
}

/*==============================================================================================
	set Control Value

	control: UVC_APL_POWER_MODE -- UVC_APL_SYNCH_DELAY
	request: SET_CUR
	data:    setting data
	len:	 length of data 
==============================================================================================*/

BYTE	USBHostUVCSetControlValue(BYTE control, BYTE request, BYTE *data, BYTE len)
{
	BYTE	errorCode, rc;
    DWORD   byteCount;
	BYTE	info, size;
	WORD	type;

	// Check control range
	if(control > UVC_APL_SYNCH_DELAY) return UVC_INVALID;

	// check request, request is SET_CUR only.
	if(request != SET_CUR) return UVC_INVALID;

	// check support in unit or terminal
	if(control > UVC_APL_ERROR_CODE && control < UVC_APL_INPUT_SELECT){
		// CT
		if((CT_bmControls & (1 << (control -UVC_APL_SCANNING))) == 0){
			// don't support control
			return UVC_UNSUPPORTED_CONTROL;
		}
	}else if(control > UVC_APL_INPUT_SELECT && control < UVC_APL_PROBE){
		// PU
		if((PU_bmControls & (1 << (control -UVC_APL_BRIGHTNESS))) == 0){
			// don't support control
			return UVC_UNSUPPORTED_CONTROL;
		}
	}

	// check support SET request
	if((UVC_Control[control].bRequest & 1) == 0){
		rc = USBHostUVCGetControlValue(control, GET_INFO, &info, &size, &type);
		if((rc != UVC_SUCCESS) || !(info & 0x20))
			return UVC_UNSUPPORTED_REQUEST;
	}

	// Set data to device
	if((errorCode = USBHostIssueDeviceRequest(
		deviceAddress,	    // address
		USB_SETUP_HOST_TO_DEVICE|USB_SETUP_TYPE_CLASS|USB_SETUP_RECIPIENT_INTERFACE, // bmRequestType
		request,					        // bRequest
		UVC_Control[control].wValue,        // wValue
		wUT_ID[UVC_Control[control].wIndex],// wIndex
		(WORD)len,                          // wLength
		data,                               // data
		USB_DEVICE_REQUEST_SET,             // direction
		driverId                            // clientDriverID
		            )) == USB_SUCCESS){
        while(USBHostTransferIsComplete( deviceAddress, 0, &errorCode, &byteCount ) != TRUE){
            USBHostTasks();
        }
    }

	if(errorCode != USB_SUCCESS){
		USBHostClearEndpointErrors( deviceAddress, 0 );
		#ifdef DEBUG_MODE
		    putstr("Control Set request error!\r\n");
		#endif
		return UVC_FAIL;
	}

	// request success
	return UVC_SUCCESS;

}


/*=============================================================================================
	set image format and resolution for frame capturing 
	(VideoStreaming Interface Probe negotiation) 
==============================================================================================*/

BYTE USBHostUVCSetFrame(BYTE format_number, WORD width, WORD height) 
{
	WORD	frame_id;
	VS_PROBE_DATA  data;
	BYTE 	rc, len;
	WORD	type;
    DWORD   byteCount;
    BYTE    errorCode;

    // Video Streaming Interface must be set to alternate zero, before probe SET_CUR request.
    //  because Logitech webcam C-525 hunging up. Other webcam is default setting is alternate zero.
	// Select zero bandwidth interface
	if((errorCode = USBHostIssueDeviceRequest(
		deviceAddress,              // address
		USB_SETUP_HOST_TO_DEVICE|USB_SETUP_TYPE_STANDARD|USB_SETUP_RECIPIENT_INTERFACE, // bmRequestType
		USB_REQUEST_SET_INTERFACE,  // bRequest   SET_INTERFACE
		0,                          // wValue     AlternateNumber
		InterfaceNum[1],            // wIndex     VideoStreaming Interface Number
		0,                          // wLength    0
		NULL,           		    // data       no data
		USB_DEVICE_REQUEST_SET,     // direction  Host to Device
		driverId                    // clientDriverID
		            )) == USB_SUCCESS){
        while(USBHostTransferIsComplete( deviceAddress, 0, &errorCode, &byteCount ) != TRUE){
            USBHostTasks();
        }
    }

	if(errorCode != USB_SUCCESS){
		USBHostClearEndpointErrors( deviceAddress, 0 );
		#ifdef DEBUG_MODE
		putstr("SET_INTERFACE request error!\r\n");
		#endif
		return UVC_FAIL;
	}

	// check format
	if(format_number >= NumFormats /*** ||
		VSFormat[format_number].format != FMT_YUY2 ***/ ){
		return UVC_UNSUPPORTED_FORMAT;
	}

	// check resolution
	for(frame_id = 0; frame_id < VSFormat[format_number].bNumFrames; frame_id++){
		if(VSFormat[format_number].frame[frame_id].wWidth == width && 
			VSFormat[format_number].frame[frame_id].wHeight == height){
			break;
		}
	}
	if(frame_id >= VSFormat[format_number].bNumFrames) return UVC_UNSUPPORTED_RESOLUTION;

	// set probe data
	memset(&data, 0, sizeof(data));
	data.bFormatIndex = VSFormat[format_number].bFormatIndex;
	data.bFrameIndex = VSFormat[format_number].frame[frame_id].bFrameIndex;

	data.bmHint=1;
	data.dwFrameInterval = VSFormat[format_number].frame[frame_id].dwFrameInterval;
	// data.dwFrameInterval = 1000000L; // 1000x0.msec=100msec=10fps
	data.wCompQuality = 50000;	// select minimum compression

	// set probe
	len = UVC_version >= 0x0110 ? 34: 26;
	rc = USBHostUVCSetControlValue(UVC_APL_PROBE, SET_CUR, (BYTE *)&data, len);
	if(rc != UVC_SUCCESS){
		#ifdef DEBUG_MODE
		    putstr("Probe SET_CUR error!!\r\n");
		#endif
		return UVC_FAIL; 
	}

	// get probe
	memset(&data, 0, sizeof(data));
	rc = USBHostUVCGetControlValue(UVC_APL_PROBE, GET_CUR, (BYTE *)&data, &len, &type);
	if(rc != USB_SUCCESS){
		#ifdef DEBUG_MODE
		putstr("Probe GET_CUR error!!\r\n");
		#endif
		return UVC_FAIL;
	}

	#ifdef DEBUG_MODE
	{
		char pbuf[12];
		putstr("VideoStreaming Interface Probe GET result.\r\n");
		putstr("\tbmHint=");              sprintf(pbuf, "0x%04x\r\n", data.bmHint); putstr(pbuf);
		putstr("\tbFormatIndex=");        sprintf(pbuf, "0x%02x\r\n", data.bFormatIndex); putstr(pbuf);
		putstr("\tbFrameIndex=");         sprintf(pbuf, "0x%02x\r\n", data.bFrameIndex); putstr(pbuf);
		putstr("\tdwFrameInterval=");     sprintf(pbuf, "%ld\r\n",    data.dwFrameInterval); putstr(pbuf);
		putstr("\twKeyFrameRate=");       sprintf(pbuf, "0x%02x\r\n", data.wKeyFrameRate); putstr(pbuf);
		putstr("\twPFrameRate=");         sprintf(pbuf, "0x%02x\r\n", data.wPFrameRate); putstr(pbuf);
		putstr("\twCompQuality=");        sprintf(pbuf, "0x%02x\r\n", data.wCompQuality); putstr(pbuf);
		putstr("\twCompWindowSize=");     sprintf(pbuf, "0x%02x\r\n", data.wCompWindowSize); putstr(pbuf);
		putstr("\twDelay=");              sprintf(pbuf, "0x%02x\r\n", data.wDelay); putstr(pbuf);
		putstr("\tdwMaxVideoFrameSize="); sprintf(pbuf, "%ld\r\n",    data.dwMaxVideoFrameSize); putstr(pbuf);
		putstr("\tdwMaxPayloadTransferSize="); sprintf(pbuf, "%ld\r\n", data.dwMaxPayloadTransferSize); putstr(pbuf);
		if(UVC_version >= 0x0110){
			putstr("\tdwClockFrequency=");    sprintf(pbuf, "0x%08lx\r\n",data.dwClockFrequency); putstr(pbuf);
			putstr("\tbmFramingInfo=");       sprintf(pbuf, "0x%02x\r\n", data.bmFramingInfo); putstr(pbuf);
			putstr("\tbPreferedVersion=");    sprintf(pbuf, "0x%02x\r\n", data.bPreferedVersion); putstr(pbuf);
			putstr("\tbMinVersion=");         sprintf(pbuf, "0x%02x\r\n", data.bMinVersion); putstr(pbuf);
			putstr("\tbMaxVersion=");         sprintf(pbuf, "0x%02x\r\n", data.bMaxVersion); putstr(pbuf);
		}
	}
	#endif

	// fail negotiation ?
	if( data.bFormatIndex != VSFormat[format_number].bFormatIndex
		|| data.bFrameIndex != VSFormat[format_number].frame[frame_id].bFrameIndex ){
        #ifdef DEBUG_MODE
            putstr("Failed probe negotiation.\r\n");
        #endif
		return UVC_FAIL;
	}
	return UVC_SUCCESS;
}


/*=============================================================================================
	FrameCapturing
 
	BYTE USBHostUVCStartFrameCapturing(void);
	BYTE *USBHostUVCGetFrameCapturing(WORD *length);
	BYTE USBHostUVCStopFrameCapturing(void);
==============================================================================================*/

static	ISOCHRONOUS_DATA        isocData;
static	volatile ISOCHRONOUS_DATA_BUFFER *isocDataBuffer;
static	BYTE                    buf_index;
static	UVC_DATA_BUFFER	        *uvcData;

/*---------------------------------------------------------------------------------------------
	Start Frame Capture
---------------------------------------------------------------------------------------------*/

BYTE USBHostUVCStartFrameCapturing(void) 
{
	VS_PROBE_DATA data;
	BYTE 	      rc, len;
	WORD	      type;
	BYTE	      alt;
    BYTE          errorCode;
    DWORD         byteCount;
	#ifdef DEBUG_MODE
		char pbuf[12];
	#endif

	// get final probe data
	memset(&data, 0, sizeof(data));
	rc = USBHostUVCGetControlValue(UVC_APL_PROBE, GET_CUR, (BYTE *)&data, &len, &type);
	if(rc != USB_SUCCESS){
		#ifdef DEBUG_MODE
		putstr("Probe GET_CUR error!!\r\n");
		#endif
		USBHostClearEndpointErrors( deviceAddress, 0 );
		return UVC_FAIL;
	}

	#ifdef DEBUG_MODE
	{
		putstr("VideoStreaming Interface Probe GET result.\r\n");
		putstr("\tbmHint=");              sprintf(pbuf, "0x%04x\r\n", data.bmHint); putstr(pbuf);
		putstr("\tbFormatIndex=");        sprintf(pbuf, "0x%02x\r\n", data.bFormatIndex); putstr(pbuf);
		putstr("\tbFrameIndex=");         sprintf(pbuf, "0x%02x\r\n", data.bFrameIndex); putstr(pbuf);
		putstr("\tdwFrameInterval=");     sprintf(pbuf, "%ld\r\n",    data.dwFrameInterval); putstr(pbuf);
		putstr("\twKeyFrameRate=");       sprintf(pbuf, "0x%02x\r\n", data.wKeyFrameRate); putstr(pbuf);
		putstr("\twPFrameRate=");         sprintf(pbuf, "0x%02x\r\n", data.wPFrameRate); putstr(pbuf);
		putstr("\twCompQuality=");        sprintf(pbuf, "0x%02x\r\n", data.wCompQuality); putstr(pbuf);
		putstr("\twCompWindowSize=");     sprintf(pbuf, "0x%02x\r\n", data.wCompWindowSize); putstr(pbuf);
		putstr("\twDelay=");              sprintf(pbuf, "0x%02x\r\n", data.wDelay); putstr(pbuf);
		putstr("\tdwMaxVideoFrameSize="); sprintf(pbuf, "%ld\r\n",    data.dwMaxVideoFrameSize); putstr(pbuf);
		putstr("\tdwMaxPayloadTransferSize="); sprintf(pbuf, "%ld\r\n", data.dwMaxPayloadTransferSize); putstr(pbuf);
		if(UVC_version >= 0x0110){
			putstr("\tdwClockFrequency=");    sprintf(pbuf, "0x%08lx\r\n",data.dwClockFrequency); putstr(pbuf);
			putstr("\tbmFramingInfo=");       sprintf(pbuf, "0x%02x\r\n", data.bmFramingInfo); putstr(pbuf);
			putstr("\tbPreferedVersion=");    sprintf(pbuf, "0x%02x\r\n", data.bPreferedVersion); putstr(pbuf);
			putstr("\tbMinVersion=");         sprintf(pbuf, "0x%02x\r\n", data.bMinVersion); putstr(pbuf);
			putstr("\tbMaxVersion=");         sprintf(pbuf, "0x%02x\r\n", data.bMaxVersion); putstr(pbuf);
		}
	}
	#endif

	// Select AlternateInterface
	for(alt = NumAlternates; alt > 0; alt--){
		if((AlternateMaxPacketSize[alt] <= data.dwMaxPayloadTransferSize) && ((AlternateMaxPacketSize[alt] & 0x3) == 0)){
			 /* if(AlternateMaxPacketSize[alt] <= 800) */ break;
		}
	}
	if(alt == 0){
		#ifdef DEBUG_MODE
		putstr("Don't select AlternateInterface.\r\n");
		#endif
		return UVC_FAIL;
	}

	#ifdef DEBUG_MODE
	putstr("Select AlternateInterfaceNumber="); sprintf(pbuf, "%d\r\n", alt); putstr(pbuf);
	putstr("       packetSize="); sprintf(pbuf, "%d\r\n", AlternateMaxPacketSize[alt]); putstr(pbuf);
	#endif


	// create Isochronous Buffer
	rc = USBHostIsochronousBuffersCreate( &isocData, USB_MAX_ISOCHRONOUS_DATA_BUFFERS, AlternateMaxPacketSize[alt]);
	if(rc == FALSE){
		#ifdef DEBUG_MODE
		putstr("Not enough memory for buffer.\r\n");
		#endif
		return UVC_FAIL;
	}


	// set commit
	rc = USBHostUVCSetControlValue(UVC_APL_COMMIT, SET_CUR, (BYTE *)&data, len);
	if(rc != USB_SUCCESS){
		#ifdef DEBUG_MODE
		putstr("Commit SET_CUR error!!\r\n");
		#endif
		USBHostIsochronousBuffersDestroy( &isocData, USB_MAX_ISOCHRONOUS_DATA_BUFFERS );
		return UVC_FAIL;
	}


/*****
	// get commit result data
	memset(&data, 0, sizeof(data));
	rc = USBHostUVCGetControlValue(UVC_APL_COMMIT, GET_CUR, (BYTE *)&data, &len, &type);
	if(rc != USB_SUCCESS){
		#ifdef DEBUG_MODE
		putstr("Commit GET_CUR error!!\r\n");
		#endif
		USBHostClearEndpointErrors( deviceAddress, 0 );
		return UVC_FAIL;
	}

	#ifdef DEBUG_MODE
	{
		putstr("VideoStreaming Interface Commit GET result.\r\n");
		putstr("\tbmHint=");              sprintf(pbuf, "0x%04x\r\n", data.bmHint); putstr(pbuf);
		putstr("\tbFormatIndex=");        sprintf(pbuf, "0x%02x\r\n", data.bFormatIndex); putstr(pbuf);
		putstr("\tbFrameIndex=");         sprintf(pbuf, "0x%02x\r\n", data.bFrameIndex); putstr(pbuf);
		putstr("\tdwFrameInterval=");     sprintf(pbuf, "%ld\r\n",    data.dwFrameInterval); putstr(pbuf);
		putstr("\twKeyFrameRate=");       sprintf(pbuf, "0x%02x\r\n", data.wKeyFrameRate); putstr(pbuf);
		putstr("\twPFrameRate=");         sprintf(pbuf, "0x%02x\r\n", data.wPFrameRate); putstr(pbuf);
		putstr("\twCompQuality=");        sprintf(pbuf, "0x%02x\r\n", data.wCompQuality); putstr(pbuf);
		putstr("\twCompWindowSize=");     sprintf(pbuf, "0x%02x\r\n", data.wCompWindowSize); putstr(pbuf);
		putstr("\twDelay=");              sprintf(pbuf, "0x%02x\r\n", data.wDelay); putstr(pbuf);
		putstr("\tdwMaxVideoFrameSize="); sprintf(pbuf, "%ld\r\n",    data.dwMaxVideoFrameSize); putstr(pbuf);
		putstr("\tdwMaxPayloadTransferSize="); sprintf(pbuf, "%ld\r\n", data.dwMaxPayloadTransferSize); putstr(pbuf);
		if(UVC_version >= 0x0110){
			putstr("\tdwClockFrequency=");    sprintf(pbuf, "0x%08lx\r\n",data.dwClockFrequency); putstr(pbuf);
			putstr("\tbmFramingInfo=");       sprintf(pbuf, "0x%02x\r\n", data.bmFramingInfo); putstr(pbuf);
			putstr("\tbPreferedVersion=");    sprintf(pbuf, "0x%02x\r\n", data.bPreferedVersion); putstr(pbuf);
			putstr("\tbMinVersion=");         sprintf(pbuf, "0x%02x\r\n", data.bMinVersion); putstr(pbuf);
			putstr("\tbMaxVersion=");         sprintf(pbuf, "0x%02x\r\n", data.bMaxVersion); putstr(pbuf);
		}
	}
	#endif
******/

	// request send, Select Alternate Interface
	if((errorCode = USBHostIssueDeviceRequest(
		deviceAddress,	    // address
		USB_SETUP_HOST_TO_DEVICE|USB_SETUP_TYPE_STANDARD|USB_SETUP_RECIPIENT_INTERFACE, // bmRequestType
		USB_REQUEST_SET_INTERFACE,		    // bRequest   SET_INTERFACE
		alt,        						// wValue     AlternateNumber
		InterfaceNum[1],                    // wIndex     VideoStreaming Interface Number
		0,                                  // wLength    0
		NULL,                               // data       no data
		USB_DEVICE_REQUEST_SET,             // direction
		driverId                            // clientDriverID
		            )) == USB_SUCCESS){
        while(USBHostTransferIsComplete( deviceAddress, 0, &errorCode, &byteCount ) != TRUE){
            USBHostTasks();
        }
    }

	if(errorCode != USB_SUCCESS){
		USBHostClearEndpointErrors( deviceAddress, 0 );
		// error skip BUFFALO BSW13K08H   		
	    if(!(VendorId == 0x1e4e && ProductId == 0x0103)){
		    USBHostIsochronousBuffersDestroy( &isocData, USB_MAX_ISOCHRONOUS_DATA_BUFFERS );
		    #ifdef DEBUG_MODE
		    putstr("SET_INTERFACE request error!\r\n");
		    #endif
	        return UVC_FAIL;
        }   
	}

	// Start Isochronouse Transfer
	buf_index = 0;
	isocDataBuffer = &(isocData.buffers[buf_index]);
	TransferError_FLG    = 0;

	rc = USBHostReadIsochronous(deviceAddress, UVC_EP[1], &isocData);
	if(rc == USB_SUCCESS) return UVC_SUCCESS;

	#ifdef DEBUG_MODE
	putstr("HostReadIsochronous call error!\r\n");
	#endif
	USBHostIsochronousBuffersDestroy(&isocData, USB_MAX_ISOCHRONOUS_DATA_BUFFERS );
	return UVC_FAIL;
}

	
/*---------------------------------------------------------------------------------------------
	Do Frame Capturing ...
---------------------------------------------------------------------------------------------*/

UVC_DATA_BUFFER *USBHostUVCGetFrameCapturing(void)
{
	// wait for Received Complete
	while(isocDataBuffer->bfDataLengthValid == 0 && TransferError_FLG == 0){
		USBHostTasks();
	}

	if(TransferError_FLG){
		#ifdef DEBUG_MODE
		putstr("Isochronous read transfer error!\r\n");
		#endif
		return NULL;
	}


	// set return data (cut payload header)
	/**
	uvcData.length = isocDataBuffer->dataLength - isocDataBuffer->pBuffer[0];
	uvcData.buffer = isocDataBuffer->pBuffer + isocDataBuffer->pBuffer[0];
	uvcData.valid = &(isocDataBuffer->bfDataLengthValid);
	**/
	uvcData = (UVC_DATA_BUFFER *)isocDataBuffer;

	// update pointer to next buffer
	if(++buf_index >= USB_MAX_ISOCHRONOUS_DATA_BUFFERS) buf_index = 0;
	isocDataBuffer = &isocData.buffers[buf_index];

	// reset error flags for next receive
	TransferError_FLG = 0;

	return uvcData;

}

/*---------------------------------------------------------------------------------------------
	Stop Frame Capture
---------------------------------------------------------------------------------------------*/

BYTE USBHostUVCStopFrameCapturing(void)
{
	BYTE	errorCode;
    DWORD   byteCount;

	// stop the isochoronous data transfer
	USBHostTerminateTransfer(deviceAddress, UVC_EP[1]);
	USBHostTasks();

	// clear isocData buffers
	USBHostIsochronousBuffersDestroy( &isocData, USB_MAX_ISOCHRONOUS_DATA_BUFFERS );

	// Selcet zero bandwidth interface
	// Select Alternate Interface
	if((errorCode = USBHostIssueDeviceRequest(
		deviceAddress,              // address
		USB_SETUP_HOST_TO_DEVICE|USB_SETUP_TYPE_STANDARD|USB_SETUP_RECIPIENT_INTERFACE, // bmRequestType
		USB_REQUEST_SET_INTERFACE,  // bRequest   SET_INTERFACE
		0,                          // wValue     AlternateNumber
		InterfaceNum[1],            // wIndex     VideoStreaming Interface Number
		0,                          // wLength    0
		NULL,           		    // data       no data
		USB_DEVICE_REQUEST_SET,     // direction  Host to Device
		driverId                    // clientDriverID
		            )) == USB_SUCCESS){
        while(USBHostTransferIsComplete( deviceAddress, 0, &errorCode, &byteCount ) != TRUE){
            USBHostTasks();
        }
    }

	if(errorCode != USB_SUCCESS){
		USBHostClearEndpointErrors( deviceAddress, 0 );
		#ifdef DEBUG_MODE
		putstr("SET_INTERFACE request error!\r\n");
		#endif
		return UVC_FAIL;
	}

	return UVC_SUCCESS;
}


/*=============================================================================================
    BOOL USBHostUVCEventHandler ( BYTE address, USB_EVENT event, void *data, DWORD size )
  Parameters:
    BYTE address    - Address of device with the event
    USB_EVENT event - The bus event that occured
    void *data      - Pointer to event-specific data
    DWORD size      - Size of the event-specific data
==============================================================================================*/

BOOL USBHostUVCEventHandler ( BYTE address, USB_EVENT event, void *data, DWORD size )
{

    // Make sure it was for our device
    if ( address != deviceAddress){
        return FALSE;
    }

    // Handle specific events.
    switch (event){
	case EVENT_1MS:
	case EVENT_SOF:
	case EVENT_DATA_ISOC_READ:
    case EVENT_HUB_ATTACH:
    case EVENT_UNSUPPORTED_DEVICE:
    case EVENT_CANNOT_ENUMERATE:
    case EVENT_CLIENT_INIT_ERROR:
    case EVENT_OUT_OF_MEMORY:
    case EVENT_UNSPECIFIED_ERROR:
    case EVENT_TRANSFER:
    case EVENT_SUSPEND:
    case EVENT_RESUME:
 		return TRUE;

    case EVENT_DETACH:
        #ifdef DEBUG_MODE
        putstr("UVC: Device Detached.\r\n");
        #endif
        // Notify that application that the device has been detached.
        USB_HOST_APP_EVENT_HANDLER(deviceAddress, EVENT_UVC_DETACH, NULL, 0);
        driverInitialized   = 0;
        deviceAddress = 0;
		TransferError_FLG = 1;
        return TRUE;

    case EVENT_BUS_ERROR:
		TransferError_FLG = 1;
		return TRUE;

    default:
        break;
    }
    return FALSE;

} // USBHostUVCEventHandler


/*=============================================================================================
	parse configuration descriptor
==============================================================================================*/

static BYTE	parse_configuration_descriptor(void)
{
	int		n, i, len, format_index, frame_index, image_index;
	BYTE	*p;
	BYTE	bNumInterfaces;
	BYTE	bInterfaceNumber;
	BYTE	bInterfaceClass;
	BYTE	bInterfaceSubClass;

	p = USBHostGetCurrentConfigurationDescriptor(deviceAddress);

	format_index = 0;
	frame_index = 0;

	image_index = 0;
	bNumInterfaces = 0;
	bInterfaceNumber = 0;
	bInterfaceClass = 0;
	bInterfaceSubClass = 0;
	UVC_version = 0;

	len = (p[3] << 8) | p[2];
	n = 0;

	while(n < len){
		switch(p[n+1]){
		case 0x01: // Device Descriptor
			n += p[n];
			break;

		case 0x02: // configuration descriptor
			//putstr("\t*** ConfigurationDescriptor\r\n");
			//putstr("\tbLength=");               putdec(p[n+0]); putcrlf();
			//putstr("\tbDescriptor=0x");         puthex(p[n+1]); putcrlf();
			//putstr("\twTotalLength=");          putdec((unsigned int)(p[n+3])*256 + p[n+2]); putcrlf();
			//putstr("\tbNumInterfaces=0x");      puthex(p[n+4]); putcrlf();
			//putstr("\tbConfigurationValue=0x"); puthex(p[n+5]); putcrlf();
			//putstr("\tiConfiguration=0x");      puthex(p[n+6]); putcrlf();
			//putstr("\tbmAttributes=0x");        puthex(p[n+7]); putcrlf();
			//putstr("\tbMaxPower=0x");           puthex(p[n+8]); putcrlf();

			bNumInterfaces = p[n+4];
			n += p[n];
			break;

		case 0x04: // interface descriptor
			//putstr("\t*** InterfaceDescriptor\r\n");
			//putstr("\tbLength=");              putdec(p[n+0]); putcrlf();
			//putstr("\tbDescriptorType=0x");    puthex(p[n+1]); putcrlf();
			//putstr("\tbInterfaceNumber=0x");   puthex(p[n+2]); putcrlf();
			//putstr("\tbAlternateSetting=0x");  puthex(p[n+3]); putcrlf();
			//putstr("\tbNumEndPoints=0x");      puthex(p[n+4]); putcrlf();
			//putstr("\tbInterfaceClass=0x");    puthex(p[n+5]); putcrlf();
			//putstr("\tbInterfaceSubClass=0x"); puthex(p[n+6]); putcrlf();
			//putstr("\tbInterfaceProtocol=0x"); puthex(p[n+7]); putcrlf();
			//putstr("\tiInterface=0x");         puthex(p[n+8]); putcrlf();

			//if(bNumInterfaces == 0) return;
			//bNumInterfaces--;

			bInterfaceNumber = p[n+2];
			bInterfaceClass = p[n+5];
			bInterfaceSubClass = p[n+6];

			if(bInterfaceClass == CC_VIDEO && bInterfaceSubClass == SC_VIDEOCONTROL){
				InterfaceNum[0] = bInterfaceNumber;
				wUT_ID[0] = bInterfaceNumber;
			}
			if(bInterfaceClass == CC_VIDEO && bInterfaceSubClass == SC_VIDEOSTREAMING){
				InterfaceNum[1] = bInterfaceNumber;
				wUT_ID[1] = bInterfaceNumber;

				NumAlternates = p[n+3];  // bAlternateSetting
				if(AlternateMaxPacketSize == NULL){
					AlternateMaxPacketSize = (WORD *)malloc(sizeof(WORD));
				}else{
					AlternateMaxPacketSize = (WORD *)realloc(AlternateMaxPacketSize,
							(NumAlternates +1)*sizeof(WORD));
				}
				if(AlternateMaxPacketSize == NULL){
					#ifdef DEBUG_MODE
						putstr("Out of Memory! (AlternateMaxPacketSize)\r\n");
					#endif
					return UVC_FAIL;
				}
				AlternateMaxPacketSize[NumAlternates] = 0;
			}	
			n += p[n];
			break;

		case 0x05:	// endpoint descriptor
			//putstr("\t*** EndpointDescriptor\r\n");
			//putstr("\tbLength=");              putdec(p[n+0]); putcrlf();
			//putstr("\tbDescriptorType=0x");    puthex(p[n+1]); putcrlf();
			//putstr("\tbEndPointAddress=0x");   puthex(p[n+2]); putcrlf();
			//putstr("\tbmAttributes=0x");       puthex(p[n+3]); putcrlf();
			//putstr("\twMaxPacketSize=");       putdec((unsigned int)(p[n+5]) * 256 + p[n+4]); putcrlf();
			//putstr("\tbInterval=0x");          puthex(p[n+6]); putcrlf();

			if(bInterfaceClass == CC_VIDEO && bInterfaceSubClass == SC_VIDEOCONTROL){
				if(p[n+3] == 0x03){	// bmAttributes is Interrupt Transfer
					UVC_EP[0] = p[n+2];
				}
			}
			if(bInterfaceClass == CC_VIDEO && bInterfaceSubClass == SC_VIDEOSTREAMING){
				if(p[n+3] == 0x05){	// bmAttributes is Isochronous  Transfer
					UVC_EP[1] = p[n+2];
				}
				AlternateMaxPacketSize[NumAlternates] = ((WORD)(p[n+5]) << 8) + p[n+4];
			}
			n += p[n];
			break;

		case 0x03:	// string descriptor
			//putstr("\t*** StringDescriptor\r\n");
			n += p[n];
			break;

		case 0x06:	// device qualifier
			//putstr("\t*** DeviceQualifierDescriptor\r\n");
			n += p[n];
			break;

		case 0x07:	// other speed configuration
			//putstr("\t*** OtherSpeedConfigurationDescriptor\r\n");
			n += p[n];
			break;

		case 0x0b: // INTERFACE ASSOCIATION descriptor
			//putstr("\t*** Interface Association Descriptor\r\n");
			//putstr("\tbLength=");              putdec(p[n+0]); putcrlf();
			//putstr("\tbDescriptorType=0x");    puthex(p[n+1]); putcrlf();
			//putstr("\tbFirstInterface=0x");    puthex(p[n+2]); putcrlf();
			//putstr("\tbInterfaceCount=0x");    puthex(p[n+3]); putcrlf();
			//putstr("\tbFunctionClass=0x");     puthex(p[n+4]); putcrlf();
			//putstr("\tbFunctionSubClass=0x");  puthex(p[n+5]); putcrlf();
			//putstr("\tbFunctionProtocol=0x");  puthex(p[n+6]); putcrlf();
			//putstr("\tiFunction=0x");          puthex(p[n+7]); putcrlf();
			n += p[n];
			break;

		case 0x24:	// CS_INTERFACE
			if(bInterfaceClass == CC_VIDEO && bInterfaceSubClass == SC_VIDEOCONTROL /* 0x01 */){ // Video Control
				//putstr("\t*** Video Control Interface Descriptor\r\n");
				//putstr("\tbLength=");              putdec(p[n+0]); putcrlf();
				//putstr("\tbDescriptorType=0x");    puthex(p[n+1]); putcrlf();
				//putstr("\tbDescriptorSubType=0x"); puthex(p[n+2]);
				switch(p[n+2]){
				case 0x01:	// VC_HEADER
					//putstr(" [VC_HEADER]\r\n");			
					//putstr("\tbcdUVC=0x");           puthex(p[n+4]); puthex(p[n+3]); putcrlf();
					//putstr("\twTotalLength=");       putdec((unsigned int)(p[n+6]) * 256 + p[n+5]); putcrlf();
					//putstr("\tdwClockFrequency=0x"); puthex(p[n+10]);puthex(p[n+9]); puthex(p[n+8]); puthex(p[n+7]); putcrlf();
					//putstr("\tbInCollection=0x");    puthex(p[n+11]);putcrlf();
					//for(i = 0; i < p[n+11]; i++){
					//	putstr("\tbaInterfaceNr=0x"); puthex(p[n+12+i]); putcrlf();
					//}

					// get UVC version
					UVC_version = ((WORD)(p[n+4]) << 8) | p[n+3];
					break;

				case 0x02:	// VC_INPUT_TERMINAL
					//putstr(" [VC_INPUT_TERMINAL]\r\n");			
					//putstr("\tbTerminalID=0x");     puthex(p[n+3]); putcrlf();
					//putstr("\twTerminalType=0x");   puthex(p[n+5]); puthex(p[n+4]); putcrlf();
					//putstr("\tbAcssocTerminal=0x"); puthex(p[n+6]); putcrlf();
					//putstr("\tiTerminal=0x");       puthex(p[n+7]); putcrlf();
					if(p[n+4] != 0x01 || p[n+5] != 0x02){   // !CAMERA
						//for(i = 8; i < p[n]; i++){
						//	putstr("\t ..... =0x"); puthex(p[n+i]); putcrlf();
						//}
						break;
					}
					// CAMERA
					//putstr("\twObjectiveFocalLengthMin=0x"); puthex(p[n+9]); puthex(p[n+8]); putcrlf();
					//putstr("\twObjectiveFocalLengthMax=0x"); puthex(p[n+11]); puthex(p[n+10]); putcrlf();
					//putstr("\twOcularFocalLength=0x");       puthex(p[n+13]); puthex(p[n+12]); putcrlf();
					//putstr("\tbControlSize=0x");             puthex(p[n+14]); putcrlf();
					//putstr("\tbmControls=0x"); for(i = p[n+14]; i > 0; i--) puthex(p[n+14+i]); putcrlf();

					wUT_ID[VC_INPUT_TERMINAL] = ((int)(p[n+3]) << 8) | bInterfaceNumber;
					CT_bmControls = 0L;
					for(i = p[n+14]; i > 0; i--){
						CT_bmControls <<= 8;
						CT_bmControls |= p[n+14+i];
					}
					break;

				case 0x03:	// VC_OUTPUT_TERMINAL
					//putstr(" [VC_OUTPUT_TERMINAL]\r\n");			
					//putstr("\tbTerminalID=0x");     puthex(p[n+3]); putcrlf();
					//putstr("\twTerminalType=0x");   puthex(p[n+5]); puthex(p[n+4]); putcrlf();
					//putstr("\tbAcssocTerminal=0x"); puthex(p[n+6]); putcrlf();
					//putstr("\tbSourceID=0x");       puthex(p[n+7]); putcrlf();
					//putstr("\tiTerminal=0x");       puthex(p[n+8]); putcrlf();
					//for(i = 9; i < p[n]; i++){
					//	putstr("\t ..... =0x"); puthex(p[n+i]); putcrlf();
					//}

					wUT_ID[VC_OUTPUT_TERMINAL] = ((int)(p[n+3]) << 8) | bInterfaceNumber;
					break;

				case 0x04:	// VC_SELECTOR_UNIT
					//putstr(" [VC_SELECTOR_UNIT]\r\n");
					//putstr("\tbUnitID=0x");    puthex(p[n+3]); putcrlf();
					//putstr("\tbNrInPins=0x");  puthex(p[n+4]); putcrlf();
					//for(i = 0; i < p[n+4]; i++){
					//	putstr("\tbaSourceID=0x"); puthex(p[n+5+i]); putcrlf();
					//}
					//putstr("\tiSelector=0x");  puthex(p[n+5+p[n+4]]); putcrlf();

					wUT_ID[VC_SELECTOR_UNIT] = ((int)(p[n+3]) << 8) | bInterfaceNumber;
					break;

				case 0x05:	// VC_PROCESSING_UNIT
					//putstr(" [VC_PROCESSING_UNIT]\r\n");			
					//putstr("\tbUnitID=0x");       puthex(p[n+3]); putcrlf();
					//putstr("\tbSourceID=0x");     puthex(p[n+4]); putcrlf();
					//putstr("\twMaxMuliplier=0x"); puthex(p[n+6]); puthex(p[n+5]); putcrlf();
					//putstr("\tbControlSize=0x");  puthex(p[n+7]); putcrlf();
					//putstr("\tbmControls=0x"); for(i = p[n+7]; i > 0; i--) puthex(p[n+7+i]); putcrlf();
					//putstr("\tiProcessing=0x");   puthex(p[n+8+p[n+7]]); putcrlf();
					//if(p[n+7] >= 3) {
					//	putstr("\tbmVideoStandards=0x"); puthex(p[n+9+p[n+7]]); putcrlf();
					//}
					//for(i = 0; i < p[n+7]; i++){
					//	dump_process_bmControls(p[n+8+i],i);
					//}
					wUT_ID[VC_PROCESSING_UNIT] = ((int)(p[n+3]) << 8) | bInterfaceNumber;
					PU_bmControls = 0L;
					for(i = p[n+7]; i > 0; i--){
						PU_bmControls <<= 8;
						PU_bmControls |= p[n+7+i];
					}
					break;

				case 0x06:	// VC_EXTENSION_UNIT
					//putstr(" [VC_EXTENSION_UNIT]\r\n");
					//putstr("\tbUnitID=0x");      puthex(p[n+3]); putcrlf();
					//putstr("\tguidExtensionCode=0x"); for(i = 0; i < 16; i++); puthex(p[n+4+i]); putcrlf();
					//putstr("\tbNumControls=0x"); puthex(p[n+20]); putcrlf();
					//putstr("\tbNrInPins=0x");    puthex(p[n+21]); putcrlf();
					//for(i = 0; i < p[n+21]; i++){
					//	putstr("\tbaSourceID=0x"); puthex(p[n+22+i]); putcrlf(); 
					//}
					//putstr("\tbControlSize=0x"); puthex(p[n+22+p[n+21]]); putcrlf();
					//putstr("\tbmControls=0x"); for(i =p[n+22+p[n+21]] ; i > 0; i--) puthex(p[n+22+p[n+21]+i]); putcrlf();
					//putstr("\tiExtension=0x");   puthex(p[n+23+p[n+21]+p[n+22+p[n+21]]]); putcrlf();

					wUT_ID[VC_EXTENSION_UNIT] = ((int)(p[n+3]) << 8) | bInterfaceNumber;
					break;

				default:
					//putstr(" [VC_UNDEFINED]\r\n");
					//for(i = 3; i < p[n];i++){
					//	putstr("\t ..... =0x"); puthex(p[n+i]); putcrlf();
					//}
					break;
				}

			}else if(bInterfaceClass == CC_VIDEO && bInterfaceSubClass == SC_VIDEOSTREAMING /* 0x02*/){ // Video Streaming

				//putstr("\t*** Video Streaming Interface Descriptor\r\n");
				//putstr("\tbLength=0x");            puthex(p[n+0]); putcrlf();
				//putstr("\tbDescriptorType=0x");    puthex(p[n+1]); putcrlf();
				//putstr("\tbDescriptorSubType=0x"); puthex(p[n+2]);

				switch(p[n+2]){

				case 0x01:	// VS_INPUT_HEADER
					//putstr(" [VS_INPUT_HEADER]\r\n");			
					//putstr("\tbNumFormats=0x");      puthex(p[n+3]); putcrlf();
					//putstr("\twTotalLength=0x");     puthex(p[n+5]); puthex(p[n+4]); putcrlf();
					//putstr("\tbEndpointAddress=0x"); puthex(p[n+6]); putcrlf();
					//putstr("\tbmInfo=0x");           puthex(p[n+7]); putcrlf();
					//putstr("\tbTerminalLink=0x");    puthex(p[n+8]); putcrlf();
					//putstr("\tbStillCaptureMethod=0x"); puthex(p[n+9]); putcrlf();
					//putstr("\tbTriggerSupport=0x");  puthex(p[n+10]); putcrlf();
					//putstr("\tbTriggerUsage=0x");    puthex(p[n+11]); putcrlf();
					//putstr("\tbControlSize=0x");     puthex(p[n+12]); putcrlf();
					//for(i = 0; i < p[n+3]; i++){
					//	putstr("\tbmaControls=0x"); for(j = p[n+12]; j > 0; j--) puthex(p[n+12+j + p[n+3] * p[n+12] ]); putcrlf();
					//}

					CaptureMethod = p[n+9];
					break;

				case 0x02:	// VS_OUTPUT_HEADER
					//putstr(" [VS_OUTPUT_HEADER]\r\n");			
					//putstr("\tbNumFormats=0x");      puthex(p[n+3]); putcrlf();
					//putstr("\twTotalLength=0x");     puthex(p[n+5]); puthex(p[n+4]); putcrlf();
					//putstr("\tbEndpointAddress=0x"); puthex(p[n+6]); putcrlf();
					//putstr("\tbTerminalLink=0x");    puthex(p[n+7]); putcrlf();
					//putstr("\tbControlSize=0x");     puthex(p[n+8]); putcrlf();
					//for(i = 0; i < p[n+3]; i++){
					//	putstr("\tbmaControls=0x"); for(j = p[n+8]; j > 0; j--) puthex(p[n+8+j + p[n+3] * p[n+8] ]); putcrlf();
					//}
					break;

				case 0x03:	// VS_STILL_IMAGE_FRAME
					//putstr(" [VS_STILL_IMAGE_FRAME]\r\n");
					//putstr("\tbEndpointAddress=0x");      puthex(p[n+3]); putcrlf();
					//putstr("\tbNumImageSizePatterns=0x"); puthex(p[n+4]); putcrlf();
					//for(i = 0; i < p[n+4]; i++){
					//	putstr("\twWidth=");  putdec((unsigned int)(p[n+6+i*4]) * 256 + p[n+5+i*4]); putcrlf();
					//	putstr("\twHeight="); putdec((unsigned int)(p[n+8+i*4]) * 256 + p[n+7+i*4]); putcrlf();
					//}
					//putstr("\tbNumCompressionPattern=0x");  puthex(p[n+9+p[n+4]*4-4]); putcrlf();
					//for(i = 0; i < p[n+9+p[n+4]*4-4]; i++){
					//	putstr("\tbCompression=0x");  puthex(p[n+10+p[n+4]*4-4+i]); putcrlf();
					//}
					break;
				
				case 0x04:	// VS_FORMAT_UNCOMPRESSED
					//putstr(" [VS_FORMAT_UNCOMPRESSED]\r\n");
					//putstr("\tbFormatIndex=0x");       puthex(p[n+3]); putcrlf();
					//putstr("\tbNumFrameDescriptor=0x");puthex(p[n+4]); putcrlf();
					//putstr("\tguidFormat=0x"); for(i = 0; i < 16; i++) puthex(p[n+5+i]); putcrlf();
					//putstr("\tbBitsPerPixel=0x");      puthex(p[n+21]); putcrlf();
					//putstr("\tbDefaultFrameIndex=0x"); puthex(p[n+22]); putcrlf();
					//putstr("\tbAspectRatioX=0x");      puthex(p[n+23]); putcrlf();
					//putstr("\tbAspectRatioY=0x");      puthex(p[n+24]); putcrlf();
					//putstr("\tbmInterfaceFlags=0x");   puthex(p[n+25]); putcrlf();
					//putstr("\tbCopyProtect=0x");       puthex(p[n+26]); putcrlf();

					if(VSFormat == NULL){
						VSFormat = (UVC_FORMAT_INFO *)malloc(sizeof(UVC_FORMAT_INFO));
					}else{
						VSFormat = (UVC_FORMAT_INFO *)realloc(VSFormat,
							(NumFormats +1)* sizeof(UVC_FORMAT_INFO));
					}
					if(VSFormat == NULL){
						#ifdef DEBUG_MODE
						putstr("Out of Memory! (VSFormat)\r\n");
						#endif
						return UVC_FAIL;
					}

					VSFormat[NumFormats].bFormatIndex = p[n+3]; 					
					VSFormat[NumFormats].bNumFrames = p[n+4]; 					

					if(memcmp( p+n+5, GUID_FMT_YUY2, 16) == 0){
						VSFormat[NumFormats].format = FMT_YUY2;
					}else if(memcmp( p+n+5, GUID_FMT_NV12, 16) == 0){
						VSFormat[NumFormats].format = FMT_NV12;
					}else{
						VSFormat[NumFormats].format = FMT_UNKNOWN;
					}

					VSFormat[NumFormats].frame = (UVC_FRAME_INFO *)malloc(p[n+4] * sizeof(UVC_FRAME_INFO));
					if(VSFormat[NumFormats].frame == NULL){
						#ifdef DEBUG_MODE
						putstr("Out of Memory! (VSFormat[].frame)\r\n");
						#endif
						return UVC_FAIL;
					}
					format_index = NumFormats;
					frame_index = 0;
					NumFormats++;

					break;

				case 0x05:  // VS_FRAME_UNCOMPRESSD
					//putstr(" [VS_FRAME_UNCOMPRESSED]\r\n");
					//putstr("\tbFrameIndex=0x");        puthex(p[n+3]); putcrlf();
					//putstr("\tbmCapabilities=0x");     puthex(p[n+4]); putcrlf();
					//putstr("\twWidth="); putdec((int)(p[n+6]) * 256 + p[n+5]); putcrlf();
					//putstr("\twHeight=");putdec((int)(p[n+8]) * 256 + p[n+7]); putcrlf();
					//putstr("\tdwMinBitRate=0x"); puthex(p[n+12]);puthex(p[n+11]);puthex(p[n+10]);puthex(p[n+9]); putcrlf();
					//putstr("\tdwMaxBitRate=0x"); puthex(p[n+16]);puthex(p[n+15]);puthex(p[n+14]);puthex(p[n+13]); putcrlf();
					//putstr("\tdwMaxVideoFrameBufferSize=0x"); puthex(p[n+20]);puthex(p[n+19]);puthex(p[n+18]);puthex(p[n+17]); putcrlf();
					//putstr("\tdwDefaultFrameInterval=0x"); puthex(p[n+24]);puthex(p[n+23]);puthex(p[n+22]);puthex(p[n+21]); putcrlf();
					//putstr("\tbFrameIntervalType=0x");     puthex(p[n+25]); putcrlf();
					//for(i = 0; i < p[n+25]; i++){
					//	putstr("\tdwFrameInterval=0x");puthex(p[n+29+i*4]);puthex(p[n+28+i*4]);puthex(p[n+27+i*4]);puthex(p[n+26+i*4]); putcrlf();
					//}

					VSFormat[format_index].frame[frame_index].bFrameIndex = p[n+3];
					VSFormat[format_index].frame[frame_index].wWidth = ((WORD)p[n+6] << 8) | p[n+5];
					VSFormat[format_index].frame[frame_index].wHeight = ((WORD)p[n+8] << 8) | p[n+7];
					VSFormat[format_index].frame[frame_index].dwFrameInterval = ((DWORD)p[n+24]) << 24 | ((DWORD)p[n+23]) << 16 | ((WORD)p[n+22] << 8) | p[n+21];
					frame_index++;

					break;			

				case 0x06:	// VS_FORMAT_MJPEG
					//putstr(" [VS_FORMAT_MJPEG]\r\n");
					//putstr("\tbFormatIndex=0x");       puthex(p[n+3]); putcrlf();
					//putstr("\tbNumFrameDescriptors=0x");puthex(p[n+4]); putcrlf();
					//putstr("\tbmFlags=0x");            puthex(p[n+5]); putcrlf();
					//putstr("\tbDefaultFrameIndex=0x"); puthex(p[n+6]); putcrlf();
					//putstr("\tbAspectRatioX=0x");      puthex(p[n+7]); putcrlf();
					//putstr("\tbAspectRatioY=0x");      puthex(p[n+8]); putcrlf();
					//putstr("\tbmInterlaceFlags=0x");   puthex(p[n+9]); putcrlf();
					//putstr("\tbCopyProtect=0x");       puthex(p[n+10]); putcrlf();

					if(VSFormat == NULL){
						VSFormat = (UVC_FORMAT_INFO *)malloc(sizeof(UVC_FORMAT_INFO));
					}else{
						VSFormat = (UVC_FORMAT_INFO *)realloc(VSFormat,
							(NumFormats +1)* sizeof(UVC_FORMAT_INFO));
					}
					if(VSFormat == NULL){
						#ifdef DEBUG_MODE
						putstr("Out of Memory! (VSFormat)\r\n");
						#endif
						return UVC_FAIL;
					}

					VSFormat[NumFormats].bFormatIndex = p[n+3]; 					
					VSFormat[NumFormats].bNumFrames = p[n+4]; 					
					VSFormat[NumFormats].format = FMT_MJPEG;

					VSFormat[NumFormats].frame = (UVC_FRAME_INFO *)malloc(p[n+4] * sizeof(UVC_FRAME_INFO));
					if(VSFormat[NumFormats].frame == NULL){
						#ifdef DEBUG_MODE
						putstr("Out of Memory! (VSFormat[].frame)\r\n");
						#endif
						return UVC_FAIL;
					}

					format_index = NumFormats;
					frame_index = 0;
					NumFormats++;

					break;

				case 0x07:  // VS_FRAME_MJPEG
					//putstr(" [VS_FRAME_MJPEG]\r\n");
					//putstr("\tbFrameIndex=0x");        puthex(p[n+3]); putcrlf();
					//putstr("\tbmCapabilities=0x");     puthex(p[n+4]); putcrlf();
					//putstr("\twWidth="); putdec((int)(p[n+6]) * 256 + p[n+5]); putcrlf();
					//putstr("\twHeight=");putdec((int)(p[n+8]) * 256 + p[n+7]); putcrlf();
					//putstr("\tdwMinBitRate=0x"); puthex(p[n+12]);puthex(p[n+11]);puthex(p[n+10]);puthex(p[n+9]); putcrlf();
					//putstr("\tdwMaxBitRate=0x"); puthex(p[n+16]);puthex(p[n+15]);puthex(p[n+14]);puthex(p[n+13]); putcrlf();
					//putstr("\tdwMaxVideoFrameBufferSize=0x"); puthex(p[n+20]);puthex(p[n+19]);puthex(p[n+18]);puthex(p[n+17]); putcrlf();
					//putstr("\tdwDefaultFrameInterval=0x"); puthex(p[n+24]);puthex(p[n+23]);puthex(p[n+22]);puthex(p[n+21]); putcrlf();
					//putstr("\tbFrameIntervalType=0x");     puthex(p[n+25]); putcrlf();
					//if(p[n+25] == 0){	// Continuous frame interval
					//	putstr("\tdwMinFrameInterval=0x");puthex(p[n+29]);puthex(p[n+28]);puthex(p[n+27]);puthex(p[n+26]); putcrlf();
					//	putstr("\tdwMaxFrameInterval=0x");puthex(p[n+33]);puthex(p[n+32]);puthex(p[n+31]);puthex(p[n+30]); putcrlf();
					//	putstr("\tdwFrameIntervalStep=0x");puthex(p[n+37]);puthex(p[n+36]);puthex(p[n+35]);puthex(p[n+34]); putcrlf();
					//}else{	// Discrete frame intervals supported
					//	for(i = 0; i < p[n+25]; i++){
					//		putstr("\tdwFrameInterval=0x");puthex(p[n+29+i*4]);puthex(p[n+28+i*4]);puthex(p[n+27+i*4]);puthex(p[n+26+i*4]); putcrlf();
					//	}
					//}
					VSFormat[format_index].frame[frame_index].bFrameIndex = p[n+3];
					VSFormat[format_index].frame[frame_index].wWidth = ((WORD)p[n+6] << 8) | p[n+5];
					VSFormat[format_index].frame[frame_index].wHeight = ((WORD)p[n+8] << 8) | p[n+7];
					VSFormat[format_index].frame[frame_index].dwFrameInterval = ((DWORD)p[n+24]) << 24 | ((DWORD)p[n+23]) << 16 | ((WORD)p[n+22] << 8) | p[n+21];
					frame_index++;
					break;	

				case 0x08:	// Reserved
				case 0x09:	// Reserved
					break;

				case 0x0a:	// VS_FORMAT_MPEG2TS
					//putstr(" [VS_FORMAT_MPEG2TS]\r\n");
					//putstr("\tbFormatIndex=0x");  puthex(p[n+3]); putcrlf();
					//putstr("\tbDataOffset=0x");   puthex(p[n+4]); putcrlf();
					//putstr("\tbPacketLength=0x"); puthex(p[n+5]); putcrlf();
					//putstr("\tbStrideLength=0x"); puthex(p[n+6]); putcrlf();
					//putstr("\tguidFormat=0x"); for(i = 0; i < 16; i++) puthex(p[n+7+i]); putcrlf();
					break;

				case 0x0b:	// Reserved
					break;

				case 0x0d:	// VS_COLORFORMAT
					//putstr(" [VS_COLOR_FORMAT]\r\n");
					//putstr("\tbColorPrimaries=0x"); puthex(p[n+3]); putcrlf();
					//putstr("\tbTransferCharacteristics=0x"); puthex(p[n+4]); putcrlf();
					//putstr("\tbMatrixCoefficients=0x"); puthex(p[n+5]); putcrlf();
					break;

				case 0x0e:	// Reserved
				case 0x0f:	// Reserved
					break;

				case 0x10:	// VS_FORMAT_FRAME_BASED
					//putstr(" [VS_FORMAT_FRAME_BASED]\r\n");
					//putstr("\tbFormatIndex=0x");    puthex(p[n+3]); putcrlf();
					//putstr("\tbNumFrameDescriptors=0x");puthex(p[n+4]); putcrlf();
					//putstr("\tguidFormat=0x"); for(i = 0; i < 16; i++) puthex(p[n+5+i]); putcrlf();
					//putstr("\tbBitsPerPixel=0x");      puthex(p[n+21]); putcrlf();
					//putstr("\tbDefaultFrameIndex=0x"); puthex(p[n+22]); putcrlf();
					//putstr("\tbAspectRatioX=0x");      puthex(p[n+23]); putcrlf();
					//putstr("\tbAspectRatioY=0x");      puthex(p[n+24]); putcrlf();
					//putstr("\tbmInterlaceFlags=0x");   puthex(p[n+25]); putcrlf();
					//putstr("\tbCopyProtect=0x");       puthex(p[n+26]); putcrlf();
					//putstr("\tbVariableSize=0x");       puthex(p[n+27]); putcrlf();
					break;

				case 0x11:  // VS_FRAME_FRAME_BASED
					//putstr(" [VS_FRAME_FRAME_BASED]\r\n");
					//putstr("\tbFrameIndex=0x");        puthex(p[n+3]); putcrlf();
					//putstr("\tbmCapabilities=0x");     puthex(p[n+4]); putcrlf();
					//putstr("\twWidth="); putdec((int)(p[n+6]) * 256 + p[n+5]); putcrlf();
					//putstr("\twHeight=");putdec((int)(p[n+8]) * 256 + p[n+7]); putcrlf();
					//putstr("\tdwMinBitRate=0x"); puthex(p[n+12]);puthex(p[n+11]);puthex(p[n+10]);puthex(p[n+9]); putcrlf();
					//putstr("\tdwMaxBitRate=0x"); puthex(p[n+16]);puthex(p[n+15]);puthex(p[n+14]);puthex(p[n+13]); putcrlf();
					//putstr("\tdwDefaultFrameInterval=0x"); puthex(p[n+20]);puthex(p[n+19]);puthex(p[n+18]);puthex(p[n+17]); putcrlf();
					//putstr("\tbFrameIntervalType=0x");     puthex(p[n+21]); putcrlf();
					//putstr("\tdwBytesPerLine=0x"); puthex(p[n+25]);puthex(p[n+24]);puthex(p[n+23]);puthex(p[n+22]); putcrlf();

					//if(p[n+21] == 0){	// Continuous frame interval
					//	putstr("\tdwMinFrameInterval=0x");puthex(p[n+29]);puthex(p[n+28]);puthex(p[n+27]);puthex(p[n+26]); putcrlf();
					//	putstr("\tdwMaxFrameInterval=0x");puthex(p[n+33]);puthex(p[n+32]);puthex(p[n+31]);puthex(p[n+30]); putcrlf();
					//	putstr("\tdwFrameIntervalStep=0x");puthex(p[n+37]);puthex(p[n+36]);puthex(p[n+35]);puthex(p[n+34]); putcrlf();
					//}else{	// Discrete frame intervals supported
					//	for(i = 0; i < p[n+21]; i++){
					//		putstr("\tdwFrameInterval=0x");puthex(p[n+29+i*4]);puthex(p[n+28+i*4]);puthex(p[n+27+i*4]);puthex(p[n+26+i*4]); putcrlf();
					//	}
					//}
					break;


				case 0x12:	// VS_FORMAT_STREAM_BASED
					//putstr(" [VS_FORMAT_STREAM_BASED]\r\n");
					//putstr("\tbFormatIndex=0x");    puthex(p[n+3]); putcrlf();
					//putstr("\tguidFormat=0x"); for(i = 0; i < 16; i++) puthex(p[n+4+i]); putcrlf();
					//putstr("\tdwPacketLength=0x"); puthex(p[n+23]);puthex(p[n+22]);puthex(p[n+21]);puthex(p[n+20]); putcrlf();
					break;

				default:
					//putstr(" [VS_UNDEFINED]\r\n");
					//for(i = 3; i < p[n];i++){
					//	putstr("\t ..... =0x"); puthex(p[n+i]); putcrlf();
					//}
					break;
				}

			}else{
				//putstr("\t*** Video Unknown Interface Descriptor\r\n");
				//putstr("\tbLength=0x");         puthex(p[n+0]); putcrlf();
				//putstr("\tbDescriptorType=0x"); puthex(p[n+1]); putcrlf();
				//for(i = 2; i < p[n];i++){
				//	putstr("\t ..... =0x"); puthex(p[n+i]); putcrlf();
				//}
			}

			n += p[n];
			break;

		case 0x25:	// Class Specific ENDPOINT
			//putstr("\t*** CS EndpointDescriptor\r\n");
			//putstr("\tbLength=0x");            puthex(p[n+0]); putcrlf();
			//putstr("\tbDescriptorType=0x");    puthex(p[n+1]); putcrlf();
			//putstr("\tbDescriptorSubType=0x"); puthex(p[n+2]); putcrlf();
			//putstr("\twMaxTransferSize=0x");   puthex(p[n+4]); puthex(p[n+3]); putcrlf();

			n += p[n];
			break;


		default:
			//putstr("\t*** UnknownDescriptor\r\n");
			//putstr("\tbLength=0x");         puthex(p[n+0]); putcrlf();
			//putstr("\tbDescriptorType=0x"); puthex(p[n+1]); putcrlf();
			//for(i = 2; i < p[n];i++){
			//	putstr("\t ..... =0x"); puthex(p[n+i]); putcrlf();
			//}

			n+= p[n];
			break;
		}
	}


	return UVC_SUCCESS;
}


/*** end of usb_host_uvc.c **************************************************************/
