/**************************************************************************

	s1d13781.c

	LCD controler S1D13781 control module

    External Power
      EPSON LCDC J4-3,4 3.3V 
      EPSON LCDC J4-1,2 GND

	2013/7/16  S.Suwa http://www.suwa-koubou.jp	
***************************************************************************/

/** INCLUDES **************************************************************/
#include "GenericTypeDefs.h"
#include "HardwareProfile.h"
#include "s1d13781.h"
#include "rtc.h"

#include "font.h"		// 5*7dot ASCII fonts

#ifdef USE_DUMP_ROUTINES
	#include "pic_print.h"
#endif


//===============================================================================
//  Register definitions generated by 13781CFG.EXE (Build 30)
//  (C)SEIKO EPSON CORPORATION 2009. All rights reserved.
//
//  This file establishes name/value pairs for all the registers,
//  which is used by the API, applications, and display drivers.
//===============================================================================

#define REG00_REV_CODE       0x00     // Revision Code Register [READONLY]
#define REG02_PROD_CODE      0x02     // Product Code Register [READONLY]
#define REG04_POWER_SAVE     0x04     // Power Save Register
#define REG06_RESET          0x06     // Software Reset Register
#define REG10_PLL_0          0x10     // PLL Setting Register 0
#define REG12_PLL_1          0x12     // PLL Setting Register 1
#define REG14_PLL_2          0x14     // PLL Setting Register 2
#define REG16_INTCLK         0x16     // Internal Clock Configuration Register
#define REG18_PLL_3          0x18     // PLL Setting Hidden Register 0 [RESERVED]
#define REG1A_PLL_4          0x1A     // PLL Setting Hidden Register 1 [RESERVED]
#define REG1C_PLL_5          0x1C     // PLL Setting Hidden Register 2 [RESERVED]
#define REG20_PANEL_SET      0x20     // Panel Setting Register
#define REG22_DISP_SET       0x22     // Display Setting Register
#define REG24_HDISP          0x24     // Horizontal Display Width Register
#define REG26_HNDP           0x26     // Horizontal Non-Display Period Register
#define REG28_VDISP          0x28     // Vertical Display Height Register
#define REG2A_VNDP           0x2A     // Vertical Non-Display Period Register
#define REG2C_HSW            0x2C     // HS Pulse Width Register
#define REG2E_HPS            0x2E     // HS Pulse Start Position Register
#define REG30_VSW            0x30     // VS Pulse Width Register
#define REG32_VPS            0x32     // VS Pulse Start Position Register
#define REG34_LINE_COUNT     0x34     // TE Line Count Register
#define REG40_MAIN_SET       0x40     // Main Layer Setting Register
#define REG42_MAIN_SADDR_0   0x42     // Main Layer Start Address Register 0
#define REG44_MAIN_SADDR_1   0x44     // Main Layer Start Address Register 1
#define REG46_MAIN_WIDTH     0x46     // Main Layer Width Register [READONLY]
#define REG48_MAIN_HEIGHT    0x48     // Main Layer Height Register [READONLY]
#define REG50_PIP_SET        0x50     // PIP Layer Setting Register
#define REG52_PIP_SADDR_0    0x52     // PIP Layer Start Address Register 0
#define REG54_PIP_SADDR_1    0x54     // PIP Layer Start Address Register 1
#define REG56_PIP_WIDTH      0x56     // PIP Layer Width Register
#define REG58_PIP_HEIGHT     0x58     // PIP Layer Height Register
#define REG5A_PIP_XSTART     0x5A     // PIP Layer X Start Position Register
#define REG5C_PIP_YSTART     0x5C     // PIP Layer Y Start Position Register
#define REG60_PIP_EN         0x60     // PIP Enable Register
#define REG62_ALPHA          0x62     // Alpha Blending Register
#define REG64_TRANS          0x64     // Transparency Register
#define REG66_KEY_0          0x66     // Key Color Register 0
#define REG68_KEY_1          0x68     // Key Color Register 1
#define REG80_BLT_CTRL_0     0x80     // BitBLT Control Register 0 [WRITEONLY]
#define REG82_BLT_CTRL_1     0x82     // BitBLT Control Register 1
#define REG84_BLT_STATUS     0x84     // BitBLT Status Register [READONLY]
#define REG86_BLT_CMD        0x86     // BitBLT Command Register
#define REG88_BLT_SSADDR_0   0x88     // BitBLT Source Start Address Register 0
#define REG8A_BLT_SSADDR_1   0x8A     // BitBLT Source Start Address Register 1
#define REG8C_BLT_DSADDR_0   0x8C     // BitBLT Destination Start Address Register 0
#define REG8E_BLT_DSADDR_1   0x8E     // BitBLT Destination Start Address Register 1
#define REG90_BLT_RECTOFFSET 0x90     // BitBLT Rectangle Offset Register
#define REG92_BLT_WIDTH      0x92     // BitBLT Width Register
#define REG94_BLT_HEIGHT     0x94     // BitBLT Height Register
#define REG96_BLT_BGCOLOR_0  0x96     // BitBLT Background Color Register 0
#define REG98_BLT_BGCOLOR_1  0x98     // BitBLT Background Color Register 1
#define REG9A_BLT_FGCOLOR_0  0x9A     // BitBLT Foreground Color Register 0
#define REG9C_BLT_FGCOLOR_1  0x9C     // BitBLT Foreground Color Register 1

#define REGD0_GPIO_CONFIG    0xD0     // GPIO Configuration Register
#define REGD2_GPIO_STATUS    0xD2     // GPIO Status / Control Register
#define REGD4_GPIO_PULLDOWN  0xD4     // GPIO Pull-Down Control Register

#define REGFLAG_BASE         0xFC     // Special reserved flags beyond this point
#define REGFLAG_DELAY        0xFC     // PLL Register Programming Delay (in us)
#define REGFLAG_OFF_DELAY    0xFD     // LCD Panel Power Off Delay (in ms)
#define REGFLAG_ON_DELAY     0xFE     // LCD Panel Power On Delay (in ms)
#define REGFLAG_END_OF_TABLE 0xFF     // End of Registers Marker


/*===============================================================================
**  Generic Header information generated by 13781CFG.EXE (Build 33)
**  (C)SEIKO EPSON CORPORATION 2009. All rights reserved.
**
**  DEVICES   [*ON]     WxHxBPP DESCRIPTION                                      
**  --------------- ----------- -------------------------------------------------
** *Display          320x240x24 Active TFT
** 
**  DIMENSIONS          WxHxBPP STRIDE START   SADDR                             
**  --------------- ----------- ------ ------- ----------------------------------
** *Main Window      320x240x24    960 N/A     00000000h
** *PIP Window         64x64x24    192 160,0   00038400h
** 
**  CLOCKS     FREQ        SOURCE                                                
**  ---------- ----------- ------------------------------------------------------
**  SYSCLK      54.000 MHz PLL
**  PCLK         6.750 MHz SYSCLK/8
**
**  This file defines the configuration environment and registers,
**  which can be used by any software, such as display drivers.
**
**  Note: If you transfer this file to any non-PC system, use ASCII
**  mode (not BINARY) to maintain system-specific line terminators.
**===============================================================================*/

// This define for TOP-Way LCD panel LMT035KDH03(QVGA panel)
#define S1D_13781
#define S1D_DISPLAY_WIDTH               320
#define S1D_DISPLAY_HEIGHT              240
#define S1D_DISPLAY_BPP                 24
#define S1D_DISPLAY_SCANLINE_BYTES      960
#define S1D_DISPLAY_FRAMEPORTGTE          0
#define S1D_DISPLAY_PCLK                6750000L
#define S1D_PHYSICAL_REG_ADDR           0x00060800L
#define S1D_PHYSICAL_VMEM_ADDR          0x00000000L
#define S1D_PHYSICAL_REG_SIZE           593L
#define S1D_PHYSICAL_VMEM_SIZE          393216L
#define S1D_PHYSICAL_VMEM_REQUIRED      230400L
#define S1D_PALETTE_SIZE                256
#define S1D_POWER_DELAY_OFF             0
#define S1D_POWER_DELAY_ON              0
#define S1D_HWBLT
#define S1D_SWBLT
#define S1D_REGRESERVED                 0x00FC
#define S1D_REGDELAYPLL                 0x00FC
#define S1D_REGDELAYOFF                 0x00FD
#define S1D_REGDELAYON                  0x00FE


/** DEFINITION ************************************************************/
#ifdef USE_SPI_INTERFACE
	#if defined(__32MX220F032B__)
		#define	CS_ON()		LATBCLR=0x8000
		#define	CS_OFF()	LATBSET=0x8000
	#elif defined(__32MX695F512H__)
		#define	CS_ON()		LATDCLR=0x0010
		#define	CS_OFF()	LATDSET=0x0010
	#else	// PIC24F64,PIC24F256
		#define	CS_ON()		LATBbits.LATB15=0
		#define	CS_OFF()	LATBbits.LATB15=1
	#endif

	#if defined(__32MX695F512H__)
		#define	SPInBUF			SPI3BUF
		#define	SPInSTATbits	SPI3STATbits
	#else
		#define	SPInBUF			SPI1BUF
		#define	SPInSTATbits	SPI1STATbits
	#endif

	#define	CMD_RB	0xc0	// read byte
	#define	CMD_RW	0xc8	// read word
	#define	CMD_WB	0x80	// write byte
	#define	CMD_WW	0x88	// write word
#endif	// USE_SPI_INTERFACE

#ifdef	USE_INDIRECT8_INTERFACE
#endif


/** VARIABLE **************************************************************/
// This define for TOP-Way LCD panel LMT035KDH03(QVGA panel)
// Initial Setting Value: 
//       Use MAIN,PIP 2 Layer, QVGA(320x240,RGB565)/Layer
struct {
	unsigned short	idx;
	unsigned short	val;
} lcdc_reg_val[] = {

	// chip reset
    { REG06_RESET,        0x0100 }, // software reset
    { S1D_REGDELAYON,     0x2710 }, // LCD Panel Power On Delay (in ms)

	// clock setting
    { REG04_POWER_SAVE,   0x0000 }, // set PM0
    { REG10_PLL_0,        0x0000 }, // PLL
    { REG12_PLL_1,        0x000B }, // PLL Setting Register 1                
    { REG14_PLL_2,        0x0020 }, // PLL Setting Register 2                
    { REG10_PLL_0,        0x0001 }, // PLL Setting Register 0                
    { S1D_REGDELAYPLL,    0x09C4 }, // PLL Register Programming Delay (in us)
    { REG16_INTCLK,       0x0008 }, // Internal Clock Configuration Register
    { REG04_POWER_SAVE,   0x0002 }, // reset PM1

	// panel setting
    { REG20_PANEL_SET,    0x006F }, // panel set TFT24
    { REG22_DISP_SET,     0x0001 }, // display setting panel on
    { REG24_HDISP,        0x0028 }, // horizontal width 0x28(40) * 8 = 320
    { REG26_HNDP,         0x0066 }, // horizontal non-display period is 0x66
    { REG28_VDISP,        0x00F0 }, // vertical height is 0xf0(240)
    { REG2A_VNDP,         0x0021 }, // vertical non-display period is 0x21
    { REG2C_HSW,          0x0090 }, // HS pulse width            
    { REG2E_HPS,          0x0010 }, // HS pulse start position     
    { REG30_VSW,          0x0082 }, // VS Pulse width           
    { REG32_VPS,          0x0012 }, // VS pulse start position  

	// layer setting
	{ REG40_MAIN_SET,	  0x0001 }, // main layer set RGB565
	{ REG42_MAIN_SADDR_0, 0x0000 }, // main VRAM start address 0x000000
	{ REG44_MAIN_SADDR_1, 0x0000 },
	{ REG50_PIP_SET,      0x0001 }, // PIP layer set RGB565
	{ REG52_PIP_SADDR_0,  0x5800 }, // PIP VRAM start address 0x025800
	{ REG54_PIP_SADDR_1,  0x0002 },
	{ REG56_PIP_WIDTH,    0x0140 }, // PIP width is 320
	{ REG58_PIP_HEIGHT,   0x00f0 }, // PIP height is 240
	{ REG5A_PIP_XSTART,   0x0000 }, // PIP X Start is 0, PIP window size is same size Main window.
	{ REG5C_PIP_YSTART,   0x0000 }, // PIP Y Start is 0
	{ REG60_PIP_EN,       0x0000 }, // PIP disable at initial

	// alpha blending
	{ REG62_ALPHA,        0x0040 }, // PIP alpha blending default value
	{ REG64_TRANS,        0x0000 }, // transparency 
	{ REG66_KEY_0,        0x0000 }, // key color  0
	{ REG68_KEY_1,        0x0000 }, // key color  1
 };

#ifdef USE_DUMP_ROUTINES
	// data transfer work buffer
	unsigned short	word_buf[128];
#endif

// for Drive Recorder See. image.c
extern	unsigned char vramBuffer[320*2*160];
#define	byte_buf	vramBuffer
//unsigned char	byte_buf[320*2*16];	// QVGA_WIDTH * RGB565 * 16Lines
 
/** FUNCTION PROTOTYPE ****************************************************/
void	lcdc_initialize(void);
void	select_display_layer(int layer);
void	set_pip_transparency(int sw);
void	lcd_clear(int layer, unsigned short color);
void    lcd_putstr(BYTE layer, LOCATE x, LOCATE y, WORD fore_color, WORD back_color, BYTE *str);
void    lcd_putstr_wide(BYTE layer, LOCATE x, LOCATE y, WORD fore_color, WORD back_color, BYTE *str);

#ifdef USE_DUMP_ROUTINES
	void	register_byte_dump(void);
	void	register_word_dump(void);
	void	vram_dump(int layer);
#endif

unsigned char read_byte(long addr);
void          read_byte_block(long addr, unsigned char *buf, unsigned int size);
void          write_byte(long addr, unsigned char d);
void          write_byte_block(long addr, unsigned char *buf, unsigned int size);

unsigned short read_word(long addr);
void           read_word_block(long addr, unsigned short *buf, unsigned int size);
void           write_word(long addr, unsigned short d);
void           write_word_block(long addr, unsigned short *buf, unsigned int size);

#ifdef USE_SPI_INTERFACE
	void            SpiSend(unsigned char send_byte);
	unsigned char	SpiSendRec(unsigned char send_byte);
	unsigned char	SpiRec(void);
#endif

/*=========================================================================
	lcdc initialize
==========================================================================*/

void	lcdc_initialize(void)
{
	int	n;


	for(n = 0; n < 2; n++){
		write_word(REG_ADR + (long)(lcdc_reg_val[n].idx), lcdc_reg_val[n].val);
	}

	// wait for Chip Reset Time
	delay_ms(10);

	for(n = 2; n < sizeof(lcdc_reg_val)/sizeof(lcdc_reg_val[0]); n++){
		write_word(REG_ADR + (long)(lcdc_reg_val[n].idx), lcdc_reg_val[n].val);
	}

	lcd_clear(MAIN_LAYER, cBLACK);
	lcd_clear(PIP_LAYER, cBLACK);


}

/*-------------------------------------------------------------------------
	vram testbar write
--------------------------------------------------------------------------*/

unsigned short wRGB565[] = {
    0b0000000000000000,	// black
    0b1111100000000000,	// red
    0b0000011111100000, // green
    0b1111111111100000, // yellow
    0b0000000000011111, // blue 
    0b1111100000011111, // magenta
    0b0000011111111111, // cyan
    0b1111111111111111, // white
	};
	
void	vram_testbar_write(int layer)
{
	int	h, v, n;
	long adr;
	unsigned short d;

	adr = layer == MAIN_LAYER ? MAIN_VRAM_ADR: PIP_VRAM_ADR;
	n = 0;
	for(v = 0; v < 240; v++){
		for(h = 0; h < 320; h++){
			d = wRGB565[h/40];
			byte_buf[n++] = d & 0xff;
			byte_buf[n++] = (d >> 8) & 0xff;
			if(n >= 512){
				write_byte_block(adr, byte_buf, n);
				adr += n;
				n = 0;
			}
		}
	}

	if(n){
		write_byte_block(adr, byte_buf, n);
	}		

}

/*-------------------------------------------------------------------------
	select display layer 
--------------------------------------------------------------------------*/

void	select_display_layer(int layer)
{
	if(layer == PIP_LAYER){
		write_word(REG_ADR + REG60_PIP_EN, 0x0001); // PIP normal display (on)
	}else{
		write_word(REG_ADR + REG60_PIP_EN, 0x0000); // PIP blank display (off) 
	}
}

/*-------------------------------------------------------------------------
	pip layer transparency toggle switch  
--------------------------------------------------------------------------*/

void	set_pip_transparency(int sw)
{
	if(sw){
		write_word(REG_ADR + REG64_TRANS, 0x0001); 
	}else{
		write_word(REG_ADR + REG64_TRANS, 0x0000); 
	}
}


/*-------------------------------------------------------------------------
	pip layer blinking toggle switch  
--------------------------------------------------------------------------*/

void	set_pip_blinking(int sw)
{
	if(sw){
		write_word(REG_ADR + REG60_PIP_EN, 0x4002);	// 32 frame interval blink
	}else{
		write_word(REG_ADR + REG60_PIP_EN, 0x0000);
	}
}

/*-------------------------------------------------------------------------
	clear layer (fill color)  
--------------------------------------------------------------------------*/

void	lcd_clear(int layer, unsigned short color)
{
	long	n, addr;

	addr = (layer == MAIN_LAYER) ? MAIN_VRAM_ADR: PIP_VRAM_ADR;

	for(n = 0; n < QVGA_SIZE; n++, addr+=2){
		write_word(addr, color);
	}		
}


/*---------------------------------------------------------------------------------------------
	void lcd_putstr(layer, x, y, fore_color, back_color, *str)
---------------------------------------------------------------------------------------------*/

void lcd_putstr(BYTE layer, LOCATE x, LOCATE y, WORD fore_color, WORD back_color, BYTE *str)
{
	WORD	*bf;
	int		sx, sy, len, p, mask, code;
	DWORD	addr;


	for(sy = 0; sy < 8; sy++){
		bf = (WORD *)(byte_buf + (sy * 640));
		mask = (1 << sy);
		for(p = 0; str[p]; p++){
			code = str[p] - ' ';
			for(sx = 0; sx < 6; sx++){
				*bf++ = (font[code][sx] & mask) ? fore_color: back_color;
			}
		}
	}

	len = strlen(str);
	addr = (layer == MAIN_LAYER)? MAIN_VRAM_ADR: PIP_VRAM_ADR;
	addr += (y * 8 * 640 + x * 6 * 2 +4); // 4 is fixed left offset. 
	for(sy = 0; sy < 8; sy++){
		write_byte_block(addr + sy * 640, byte_buf + sy * 640, len * 6 * 2);
	}
}

/*---------------------------------------------------------------------------------------------
	void lcd_putstr_wide(layer, x, y, fore_color, back_color, *str)
---------------------------------------------------------------------------------------------*/

void lcd_putstr_wide(BYTE layer, LOCATE x, LOCATE y, WORD fore_color, WORD back_color, BYTE *str)
{
	WORD	*bf;
	int		sx, sy, len, p, mask, code;
	DWORD	addr;


	for(sy = 0; sy < 8; sy++){
		bf = (WORD *)(byte_buf + (sy * 2 * 640));
		mask = (1 << sy);
		for(p = 0; str[p]; p++){
			code = str[p] - ' ';
			for(sx = 0; sx < 6; sx++){
				*bf = *(bf +1) = *(bf + 320) = *(bf + 321) = (font[code][sx] & mask) ? fore_color: back_color;
				bf+= 2;
			}
		}
	}

	len = strlen(str);
	addr = (layer == MAIN_LAYER)? MAIN_VRAM_ADR: PIP_VRAM_ADR;
	addr += (y * 8 * 640 + x * 6 * 2 +4); // 4 is fixed left offset. 
	for(sy = 0; sy < (8 * 2); sy++){
		write_byte_block(addr + sy * 640, byte_buf + sy * 640, len * 6 * 2 * 2);
	}
}

#ifdef USE_DUMP_ROUTINES
/*-------------------------------------------------------------------------
	lcdc register byte dump
--------------------------------------------------------------------------*/

void	register_byte_dump(void)
{
	int		v,h;

	read_byte_block(REG_ADR, byte_buf, 256);

	myprintf("00: 00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F\r\n");
	myprintf("--------------------------------------------------\r\n");
	for(v = 0; v < 16; v++){
		myprintf("%02x: ", v*16);
		for(h = 0; h < 16; h++){
			myprintf("%02x ", byte_buf[v*16+h]);
		}
		myprintf("\r\n");
	}
	myprintf("\r\n");
}


/*-------------------------------------------------------------------------
	lcdc register word dump
--------------------------------------------------------------------------*/

void	register_word_dump(void)
{
	int		v,h;

	read_word_block(REG_ADR, word_buf, 128);

	myprintf("\r\n");
	myprintf("00: 0000 0002 0004 0006 0008 000A 000C 000E\r\n");
	myprintf("------------------------------------------\r\n");
	for(v = 0; v < 16; v++){
		myprintf("%02x: ", v * 16);
		for(h = 0; h < 8; h++){
			myprintf("%04x ", word_buf[v*8+h]);
		}
		myprintf("\r\n");
	}
	myprintf("\r\n");
}


/*-------------------------------------------------------------------------
	vram byte dump
--------------------------------------------------------------------------*/

void	vram_dump(int layer)
{
	int		v,h;

	read_byte_block(layer == MAIN_LAYER ? MAIN_VRAM_ADR: PIP_VRAM_ADR, byte_buf, 256);
	
	myprintf("VRAM 00-FF\r\n");
	myprintf("00: 00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F\r\n");
	myprintf("--------------------------------------------------\r\n");
	for(v = 0; v < 16; v++){
		myprintf("%02x: ", v*16);
		for(h = 0; h < 16; h++){
			myprintf("%02x ", byte_buf[v*16+h]);
		}
		myprintf("\r\n");
	}
	myprintf("\r\n");
}

#endif // USE_DUMP_ROUTINES

#ifdef USE_INDIRECT8_INTERFACE

/*************************************************************************
	INDIRECT 8 bit mode INTERFACE
		for REPIC32SD

	RB0   CS#  27
    RB1   RD#  25
    RB2   WR#  28
    RB3   P/C#  6

@@@RD0   DB0  29
@@@RD1   DB1  30 
@@@RD2   DB2  31
@@@RD3   DB3  32
@@@RD4   DB4  33
@@@RD5   DB5  34
@@@RD6   DB6  35
@@@RD7   DB7  36

bus normal state
TRISD = 0;	PORTD output
PORTB = 0b0111;  // P/C#=0,WR#=1,RD#=1,CS#=1

Write Sequence 
begin:
	Address Byte Write
	PORTB = 0b0011;   // P/C#=0,WR#=0,RD#=1,CS#=1
	PORTD = addr[0-7]
	PORTB = 0b0010;   // P/C#=0,WR#=0,RD#=1,CS#=0
	PORTB = 0b0011;   // P/C#=0,WR#=0,RD#=1,CS#=1
	PORTD = addr[8-15]
	PORTB = 0b0010;   // P/C#=0,WR#=0,RD#=1,CS#=0
	PORTB = 0b0011;   // P/C#=0,WR#=0,RD#=1,CS#=1
	PORTD = addr[16-18]
	PORTB = 0b0010;   // P/C#=0,WR#=0,RD#=1,CS#=0
	PORTB = 0b0011;   // P/C#=0,WR#=0,RD#=1,CS#=1

	Phase Change
	PORTB = 0b1011;   // P/C#=1,WR#=0,RD#=1,CS#=1

	Data Write
	while(size--){
		PORTD = data 
		PORTB = 0b1010;   // P/C#=1,WR#=0,RD#=1,CS#=0
		PORTB = 0b1011;   // P/C#=1,WR#=0,RD#=1,CS#=1
	}
	Phase Recover
	PORTB = 0b0111;   // P/C#=0,WR#=1,RD#=1,CS#=1
end:


Read Sequence 
begin:
	Address Byte Write
	PORTB = 0b0011;   // P/C#=0,WR#=0,RD#=1,CS#=1
	PORTD = addr[0-7]
	PORTB = 0b0010;   // P/C#=0,WR#=0,RD#=1,CS#=0
	PORTB = 0b0011;   // P/C#=0,WR#=0,RD#=1,CS#=1
	PORTD = addr[8-15]
	PORTB = 0b0010;   // P/C#=0,WR#=0,RD#=1,CS#=0
	PORTB = 0b0011;   // P/C#=0,WR#=0,RD#=1,CS#=1
	PORTD = addr[16-18]
	PORTB = 0b0010;   // P/C#=0,WR#=0,RD#=1,CS#=0
	PORTB = 0b0011;   // P/C#=0,WR#=0,RD#=1,CS#=1

	Phase Change
	PORTB = 0b1101;   // P/C#=1,WR#=1,RD#=0,CS#=1
	TRISD = 0xffff;  // PORTD input 

	Data Read
	while(size--){
		PORTB = 0b1100;   // P/C#=1,WR#=1,RD#=0,CS#=0
		data = PORTD;
		PORTB = 0b1101;   // P/C#=1,WR#=1,RD#=0,CS#=1
	}

	TRISD = 0;  // PORTD output 
	Phase Recover
	PORTB = 0b0111;   // P/C#=0,WR#=1,RD#=1,CS#=1
end:


*************************************************************************** */

/*--------------------------------------------------------------------------
	lcdc read byte
-------------------------------------------------------------------------*/

unsigned char	read_byte(long addr)
{
	unsigned char data;

	// address write mode
	// CS# is always high during idle state.
	LATB = (LATB & 0xfff0)|0b0011; // P/C#=0,WR#=0,RD#=1,CS#=1

	LATD = (LATD & 0xff00)|(addr & 0xff);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1
	LATD = (LATD & 0xff00)|((addr >> 8) & 0xff);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1
	LATD = (LATD & 0xff00)|((addr >> 16) & 0xff);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1

	// data read mode
	LATB = (LATB & 0xfff0)|0b1101; // P/C#=1,WR#=1,RD#=0,CS#=1
	TRISD |= 0xff;   // RD0-RD7 input 

	LATBCLR = 0b0001; // P/C#=1,WR#=1,RD#=0,CS#=0
	data = PORTD;
	LATBSET = 0b0001; // P/C#=1,WR#=1,RD#=0,CS#=1

	// phase Recover
	TRISD &= 0xff00; // RD0-RD7 output

	return data;

}


/*--------------------------------------------------------------------------
	lcdc read byte blcok
--------------------------------------------------------------------------*/

void	read_byte_block(long addr, unsigned char *buf, unsigned int size)
{
	unsigned char data;

	// address write mode
	// CS# is always high during idle state.
	LATB = (LATB & 0xfff0)|0b0011; // P/C#=0,WR#=0,RD#=1,CS#=1

	LATD = (LATD & 0xff00)|(addr & 0xff);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1
	LATD = (LATD & 0xff00)|((addr >> 8) & 0xff);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1
	LATD = (LATD & 0xff00)|((addr >> 16) & 0xff);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1

	// data read mode
	LATB = (LATB & 0xfff0)|0b1101; // P/C#=1,WR#=1,RD#=0,CS#=1
	TRISD |= 0xff;   // RD0-RD7 input 

	while(size--){
		// data one byte read
		LATBCLR = 0b0001; // P/C#=1,WR#=1,RD#=0,CS#=0
		data = PORTD;
		LATBSET = 0b0001; // P/C#=1,WR#=1,RD#=0,CS#=1
		*buf++ = data;
	}

	// phase Recover
	TRISD &= 0xff00; // RD0-RD7 output

}


/*--------------------------------------------------------------------------
	lcdc read word
--------------------------------------------------------------------------*/

unsigned short	read_word(long addr)
{
	unsigned short dh, dl;

	// address write mode
	// CS# is always high during idle state.
	LATB = (LATB & 0xfff0)|0b0011; // P/C#=0,WR#=0,RD#=1,CS#=1

	LATD = (LATD & 0xff00)|(addr & 0xff);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1
	LATD = (LATD & 0xff00)|((addr >> 8) & 0xff);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1
	LATD = (LATD & 0xff00)|((addr >> 16) & 0xff);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1

	// data read mode
	LATB = (LATB & 0xfff0)|0b1101; // P/C#=1,WR#=1,RD#=0,CS#=1
	TRISD |= 0xff;   // RD0-RD7 input 

	// data low byte read
	LATBCLR = 0b0001; // P/C#=1,WR#=1,RD#=0,CS#=0
	dl = PORTD;
	LATBSET = 0b0001; // P/C#=1,WR#=1,RD#=0,CS#=1

	// data high byte read
	LATBCLR = 0b0001; // P/C#=1,WR#=1,RD#=0,CS#=0
	dh = PORTD;
	LATBSET = 0b0001; // P/C#=1,WR#=1,RD#=0,CS#=1

	// phase Recover
	TRISD &= 0xff00; // RD0-RD7 output

	return ((dh << 8) | dl);
}


/*--------------------------------------------------------------------------
	lcdc read word blcok
--------------------------------------------------------------------------*/

void	read_word_block(long addr, unsigned short *buf, unsigned int size)
{
	unsigned short dh, dl;

	// address write mode
	// CS# is always high during idle state.
	LATB = (LATB & 0xfff0)|0b0011; // P/C#=0,WR#=0,RD#=1,CS#=1

	LATD = (LATD & 0xff00)|(addr & 0xff);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1
	LATD = (LATD & 0xff00)|((addr >> 8) & 0xff);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1
	LATD = (LATD & 0xff00)|((addr >> 16) & 0xff);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1

	// data read mode
	LATB = (LATB & 0xfff0)|0b1101; // P/C#=1,WR#=1,RD#=0,CS#=1
	TRISD |= 0xff;   // RD0-RD7 input 

	while(size--){
		// data low byte read
		LATBCLR = 0b0001; // P/C#=1,WR#=1,RD#=0,CS#=0
		dl = PORTD;
		LATBSET = 0b0001; // P/C#=1,WR#=1,RD#=0,CS#=1

		// data high byte read
		LATBCLR = 0b0001; // P/C#=1,WR#=1,RD#=0,CS#=0
		dh = PORTD;
		LATBSET = 0b0001; // P/C#=1,WR#=1,RD#=0,CS#=1

		*buf++ = ((dh >> 8) | dl);
	}

	// phase Recover
	TRISD &= 0xff00; // RD0-RD7 output
}


/*--------------------------------------------------------------------------
	lcdc	write byte
--------------------------------------------------------------------------*/

void	write_byte(long addr, unsigned char d)
{
	// address write mode
	// CS# is always high during idle state.
	LATB = (LATB & 0xfff0)|0b0011; // P/C#=0,WR#=0,RD#=1,CS#=1

	//LATD = (LATD & 0xff00)|(addr & 0xff);
	LATD = addr;
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1
	//LATD = (LATD & 0xff00)|((addr >> 8) & 0xff);
	LATD = (addr >> 8);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1
	//LATD = (LATD & 0xff00)|((addr >> 16) & 0xff);
	LATD = (addr >> 16);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1

	// data write mode
	LATBSET = 0b1000; // P/C#=1,WR#=0,RD#=1,CS#=1

	//one byte Write
	//LATD = (LATD & 0xff00)|d;
	LATD = d;
	LATBCLR = 0b0001; // P/C#=1,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=1,WR#=0,RD#=1,CS#=1

	// We return with CS# high.

}


/*--------------------------------------------------------------------------
	lcdc	write byte block
--------------------------------------------------------------------------*/

void	write_byte_block(long addr, unsigned char *buf, unsigned int size)
{
	// address write mode
	// CS# is always high level during idle state.
	LATB = (LATB & 0xfff0)|0b0011; // P/C#=0,WR#=0,RD#=1,CS#=1

	//LATD = (LATD & 0xff00)|(addr & 0xff);
	LATD = addr;
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1
	//LATD = (LATD & 0xff00)|((addr >> 8) & 0xff);
	LATD = (addr >> 8);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1
	//LATD = (LATD & 0xff00)|((addr >> 16) & 0xff);
	LATD = (addr >> 16);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1

	// data write mode
	LATBSET = 0b1000; // P/C#=1,WR#=0,RD#=1,CS#=1

	while(size--){
		//one byte Write
		//LATD = (LATD & 0xff00)| *buf++;
		LATD = *buf++;
		LATBCLR = 0b0001; // P/C#=1,WR#=0,RD#=1,CS#=0
		LATBSET = 0b0001; // P/C#=1,WR#=0,RD#=1,CS#=1
	}

	// We return with CS# high.

}


/*--------------------------------------------------------------------------
	lcdc	write word
--------------------------------------------------------------------------*/

void	write_word(long addr, unsigned short d)
{
	// address write mode
	// CS# is always high during idle state.
	LATB = (LATB & 0xfff0)|0b0011; // P/C#=0,WR#=0,RD#=1,CS#=1

	LATD = (LATD & 0xff00)|(addr & 0xff);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1
	LATD = (LATD & 0xff00)|((addr >> 8) & 0xff);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1
	LATD = (LATD & 0xff00)|((addr >> 16) & 0xff);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1

	// data write mode
	LATBSET = 0b1000; // P/C#=1,WR#=0,RD#=1,CS#=1

	// low byte Write
	LATD = (LATD & 0xff00)|(d & 0xff);
	LATBCLR = 0b0001; // P/C#=1,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=1,WR#=0,RD#=1,CS#=1
	// high byte Write
	LATD = (LATD & 0xff00)|((d >> 8) & 0xff);
	LATBCLR = 0b0001; // P/C#=1,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=1,WR#=0,RD#=1,CS#=1

	// We return with CS# high.

}


/*-------------------------------------------------------------------------
	lcdc	write word block
-------------------------------------------------------------------------*/

void	write_word_block(long addr, unsigned short *buf, unsigned int size)
{
	unsigned short d;

	// address write mode
	// CS# is always high during idle state.
	LATB = (LATB & 0xfff0)|0b0011; // P/C#=0,WR#=0,RD#=1,CS#=1

	LATD = (LATD & 0xff00)|(addr & 0xff);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1
	LATD = (LATD & 0xff00)|((addr >> 8) & 0xff);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1
	LATD = (LATD & 0xff00)|((addr >> 16) & 0xff);
	LATBCLR = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=0
	LATBSET = 0b0001; // P/C#=0,WR#=0,RD#=1,CS#=1

	// data write mode
	LATBSET = 0b1000; // P/C#=1,WR#=0,RD#=1,CS#=1

	while(size--){
		d = *buf++;
		//low byte Write
		LATD = (LATD & 0xff00)|(d & 0xff);
		LATBCLR = 0b0001; // P/C#=1,WR#=0,RD#=1,CS#=0
		LATBSET = 0b0001; // P/C#=1,WR#=0,RD#=1,CS#=1
		//high byte Write
		LATD = (LATD & 0xff00)|((d >> 8) & 0xff);
		LATBCLR = 0b0001; // P/C#=1,WR#=0,RD#=1,CS#=0
		LATBSET = 0b0001; // P/C#=1,WR#=0,RD#=1,CS#=1
	}

}

#endif // USE_INDIRECT8_INTERFACE


#ifdef USE_SPI_INTERFACE
/*************************************************************************
	SPI INTERFACE 
*************************************************************************/
/*--------------------------------------------------------------------------
	lcdc read byte
--------------------------------------------------------------------------*/

unsigned char	read_byte(long addr)
{
	unsigned char d;

	CS_ON();
	SpiSend(CMD_RB);
	SpiSend((addr >> 16) & 0xff);
	SpiSend((addr >> 8) & 0xff);
	SpiSend(addr & 0xff);
	SpiRec();	// skip dummy byte
	d = SpiRec();
	CS_OFF();
	return d;
}

/*--------------------------------------------------------------------------
	lcdc read byte blcok
--------------------------------------------------------------------------*/

void	read_byte_block(long addr, unsigned char *buf, unsigned int size)
{
	CS_ON();
	SpiSend(CMD_RB);
	SpiSend((addr >> 16) & 0xff);
	SpiSend((addr >> 8) & 0xff);
	SpiSend(addr & 0xff);
	SpiRec(); 	// skip dummy byte
	while(size--){
		*buf++ = SpiRec();
	}
	CS_OFF();
}


/*--------------------------------------------------------------------------
	lcdc read word
--------------------------------------------------------------------------*/

unsigned short	read_word(long addr)
{
	unsigned short dh, dl;

	CS_ON();
	SpiSend(CMD_RW);
	SpiSend((addr >> 16) & 0xff);
	SpiSend((addr >> 8) & 0xff);
	SpiSend(addr & 0xff);
	SpiRec();	// skip
	SpiRec();	// skip
	dh = SpiRec();
	dl = SpiRec();
	CS_OFF();

	return (dh << 8) | dl;
}


/*--------------------------------------------------------------------------
	lcdc read word blcok
--------------------------------------------------------------------------*/

void	read_word_block(long addr, unsigned short *buf, unsigned int size)
{
	unsigned short dh, dl;

	CS_ON();
	SpiSend(CMD_RW);
	SpiSend((addr >> 16) & 0xff);
	SpiSend((addr >> 8) & 0xff);
	SpiSend(addr & 0xff);
	SpiRec(); 	// skip dummy byte
	SpiRec();	// skip dummy byte
	while(size--){
		dh = SpiRec();
		dl = SpiRec();
		*buf++ = ((dh << 8) | dl);
	}
	CS_OFF();
}


/*--------------------------------------------------------------------------
	lcdc	write byte
--------------------------------------------------------------------------*/

void	write_byte(long addr, unsigned char d)
{
	CS_ON();
	SpiSend(CMD_WB);
	SpiSend((addr >> 16) & 0xff);
	SpiSend((addr >> 8) & 0xff);
	SpiSend(addr & 0xff);
	SpiSend(d);
	CS_OFF();
}


/*--------------------------------------------------------------------------
	lcdc	write byte block
--------------------------------------------------------------------------*/

void	write_byte_block(long addr, unsigned char *buf, unsigned int size)
{
	CS_ON();
	SpiSend(CMD_WB);
	SpiSend((addr >> 16) & 0xff);
	SpiSend((addr >> 8) & 0xff);
	SpiSend(addr & 0xff);
	while(size--){
		SpiSend(*buf++);
	}
	CS_OFF();
}


/*--------------------------------------------------------------------------
	lcdc	write word
--------------------------------------------------------------------------*/

void	write_word(long addr, unsigned short d)
{
	CS_ON();
	SpiSend(CMD_WW);
	SpiSend((addr >> 16) & 0xff);
	SpiSend((addr >> 8) & 0xff);
	SpiSend(addr & 0xff);
	SpiSend((d >> 8) & 0xff);
	SpiSend(d & 0xff);
	CS_OFF();
}

/*-------------------------------------------------------------------------
	lcdc	write word block
--------------------------------------------------------------------------*/

void	write_word_block(long addr, unsigned short *buf, unsigned int size)
{
	unsigned short d;

	CS_ON();
	SpiSend(CMD_WW);
	SpiSend((addr >> 16) & 0xff);
	SpiSend((addr >> 8) & 0xff);
	SpiSend(addr & 0xff);
	while(size--){
		d = *buf++;
		SpiSend((d>> 8) & 0xff);
		SpiSend(d & 0xff);
	}	
	CS_OFF();
}


/*--------------------------------------------------------------------------
	SpiSend()	received data ignore
--------------------------------------------------------------------------*/

void	SpiSend(unsigned char send_byte)
{
	unsigned char dummy;

//myprintf("%02x ", send_byte);

	// wait until TXBUF empty
	while(SPInSTATbits.SPITBF);

	// write to TXBUF
	SPInBUF = send_byte;

	// wait until data send out
	//  (wait for data received complete)
	while(!SPInSTATbits.SPIRBF);

	dummy = SPInBUF;	// clear receive buffer 
}


/*--------------------------------------------------------------------------
	SpiSendRec()	data send and data receive 
--------------------------------------------------------------------------*/

unsigned char	SpiSendRec(unsigned char send_byte)
{
	// wait until TXBUF empty
	while(SPInSTATbits.SPITBF);

	// write to TXBUF
	SPInBUF = send_byte;

	// wait until data send out
	//  (wait for data received complete)
	while(!SPInSTATbits.SPIRBF);

	return SPInBUF;
}


/*--------------------------------------------------------------------------
	SpiRec()	data receive( dummy send ) 
--------------------------------------------------------------------------*/

unsigned char	SpiRec(void)
{
	// wait until TXBUF empty
	while(SPInSTATbits.SPITBF);

	// write to dummy data
	SPInBUF = 0xff;

	// wait until data send out
	//  (wait for data received complete)
	while(!SPInSTATbits.SPIRBF);

	return SPInBUF;
}

#endif	// USE_SPI_INTERFACE

/*** end of s1d13781.c ******************************************************/

