/*------------------------------------------------------------------------/
/  MMCv3/SDv1/SDv2 (in SPI mode) control module
/-------------------------------------------------------------------------/
/
/  Copyright (C) 2013, ChaN, all right reserved.
/
/ * This software is a free software and there is NO WARRANTY.
/ * No restriction on use. You can use, modify and redistribute it for
/   personal, non-profit or commercial products UNDER YOUR RESPONSIBILITY.
/ * Redistributions of source code must retain the above copyright notice.
/
/-------------------------------------------------------------------------*/

 
//#include <p32xxxx.h>
#include "HardwareProfile.h"
#include "diskio.h"
#include "tskchg.h"

#include "uart2.h"
#define	putstr(x)	UART2PrintString(x)
#define	inkey()		UART2IsPressed()
#define	getch()		UART2GetChar()
#define	puthex(x)	UART2PutHex(x)
#define	putch(x)	UART2PutChar(x)


#define	USE_DMA   1

#if	USE_DMA
    #define SPI_TX_DMA_CH  DMA_CHANNEL0
    #define SPI_RX_DMA_CH  DMA_CHANNEL1
    #define ISR_IPL_NUM    3   // interrupt priority, Timer(2) < DMA(3) < USB(4)	
	#define BGTskID        1   // DMA ̊ԂɎs^XN
#endif


/* Port Controls  (Platform dependent) */
#define CS_LOW()  LATGCLR = 0x0200;	/* RG9 MMC CS = L */
#define CS_HIGH() LATGSET = 0x0200;	/* RG9 MMC CS = H */

#define INS	!(PORTBbits.RB5)	/* RB5 Card detected   (yes:true, no:false, default:true) */
#define WP	 (PORTBbits.RB4)	/* RB4 Write protected (yes:true, no:false, default:false) */


#define	BRGREG(x)	(((GetPeripheralClock()/2)/(x))-1)
#define	FCLK_SLOW()	(SPI2BRG=BRGREG(200000))	//200KHz	/* Set slow clock (100k-400k) */
//#define	FCLK_FAST()	(SPI2BRG=BRGREG(40000000))	//40MHz		/* Set fast clock (depends on the CSD) */
#define	FCLK_FAST()	(SPI2BRG=BRGREG(20000000))	//20MHz		/* Set fast clock (depends on the CSD) */
#define	FCLK_MID()	(SPI2BRG=BRGREG(20000000))	//20MHz		/* Set fast clock (depends on the CSD) */


/*--------------------------------------------------------------------------

   Module Private Functions

---------------------------------------------------------------------------*/

/* Definitions for MMC/SDC command */
#define CMD0   (0)			/* GO_IDLE_STATE */
#define CMD1   (1)			/* SEND_OP_COND */
#define	CMD6   (6)			/* SWITCH FUNCTION */
#define ACMD41 (41|0x80)	/* SEND_OP_COND (SDC) */
#define CMD8   (8)			/* SEND_IF_COND */
#define CMD9   (9)			/* SEND_CSD */
#define CMD10  (10)			/* SEND_CID */
#define CMD12  (12)			/* STOP_TRANSMISSION */
#define ACMD13 (13|0x80)	/* SD_STATUS (SDC) */
#define CMD16  (16)			/* SET_BLOCKLEN */
#define CMD17  (17)			/* READ_SINGLE_BLOCK */
#define CMD18  (18)			/* READ_MULTIPLE_BLOCK */
#define CMD23  (23)			/* SET_BLOCK_COUNT */
#define ACMD23 (23|0x80)	/* SET_WR_BLK_ERASE_COUNT (SDC) */
#define CMD24  (24)			/* WRITE_BLOCK */
#define CMD25  (25)			/* WRITE_MULTIPLE_BLOCK */
#define CMD41  (41)			/* SEND_OP_COND (ACMD) */
#define CMD55  (55)			/* APP_CMD */
#define CMD58  (58)			/* READ_OCR */


static volatile
DSTATUS Stat = STA_NOINIT;	/* Disk status */

static volatile
UINT Timer1, Timer2;		/* 1000Hz decrement timer */

static
UINT CardType;


/*-----------------------------------------------------------------------*/
/* Power Control  (Platform dependent)                                   */
/*-----------------------------------------------------------------------*/
/* When the target system does not support socket power control, there   */
/* is nothing to do in these functions.                                  */

static
void power_on (void)
{
	;					/* Turn on socket power, delay >1ms (Nothing to do) */

#if USE_DMA

	// initialize SPI_TX_DMA(ch0)
    DmaChnOpen(SPI_TX_DMA_CH, DMA_CHN_PRI2, DMA_OPEN_DEFAULT);
    DmaChnSetEventControl(SPI_TX_DMA_CH, DMA_EV_START_IRQ(_SPI2_TX_IRQ));
    DmaChnSetEvEnableFlags(SPI_TX_DMA_CH, DMA_EV_BLOCK_DONE);
    DmaChnSetIntPriority(SPI_TX_DMA_CH, ISR_IPL_NUM, INT_SUB_PRIORITY_LEVEL_0);
    DmaChnIntEnable(SPI_TX_DMA_CH);

	// initialize SPI_RX_DMA(ch1)
    DmaChnOpen(SPI_RX_DMA_CH, DMA_CHN_PRI3, DMA_OPEN_DEFAULT);
    DmaChnSetEventControl(SPI_RX_DMA_CH, DMA_EV_START_IRQ(_SPI2_RX_IRQ));
    DmaChnSetEvEnableFlags(SPI_RX_DMA_CH, DMA_EV_BLOCK_DONE);
    DmaChnSetIntPriority(SPI_RX_DMA_CH, ISR_IPL_NUM, INT_SUB_PRIORITY_LEVEL_0);
    DmaChnIntEnable(SPI_RX_DMA_CH);

#endif	// USE_DMA

	// initialize SPI2
	SPI2CON=0;
	FCLK_SLOW();
	SPI2STATbits.SPIROV=0;
	//SPI2CON = 0x0001012D;	/* Enable SPI2 */
	//SPI2CON = 0x0001032D;	/* Enable SPI2 */
//#if	BUFFERED_TRANSFER
//	SPI2CONbits.ENHBUF=1;
//#endif
//	SPI2CONbits.SMP=1;
	SPI2CONbits.CKE=1;
	//SPI2CONbits.CKP=1;
	SPI2CONbits.MSTEN=1;

	// SPI2CONbits.SMP=0;
	// SPI2CONbits.CKE=0;
	// SPI2CONbits.CKP=1;

	SPI2CONbits.ON=1;

}

static
void power_off (void)
{
	SPI2CONbits.ON = 0;			/* Disable SPI1 */

	;					/* Turn off socket power (Nothing to do) */

	Stat |= STA_NOINIT;	/* Force uninitialized */
}


/*-----------------------------------------------------------------------*/
/* Transmit/Receive data to/from MMC via SPI  (Platform dependent)       */
/*-----------------------------------------------------------------------*/

static
BYTE xchg_spi (BYTE dat)
{
	BYTE	dummy;


	// wait until TXBUF empty
	while(SPI2STATbits.SPITBF);

	// wait busy
	while(SPI2STATbits.SPIRBF){dummy=SPI2BUF;}

	// data sent
	SPI2BUF = dat;

	// wait until data sent
	//  (wait for data received complete)
	while (!SPI2STATbits.SPIRBF);
	return (BYTE)SPI2BUF;

}


#if !USE_DMA

/* Alternative macro to transfer data fast */
#define XMIT_SPI_MULTI(src,cnt)	{\
	BYTE d;\
	UINT c=cnt/2;\
	const BYTE *p=src;\
	do {\
		SPI2BUF=*p++;\
		while(!SPI2STATbits.SPIRBF);\
		d=SPI2BUF;\
		SPI2BUF=*p++;\
		while(!SPI2STATbits.SPIRBF);\
		d=SPI2BUF;\
	}\
	while(--c);}


/***** Below original code didn't work in my project. S.Suwa
#define XMIT_SPI_MULTI(src,cnt)	{\
	UINT c=cnt/2;\
	const BYTE *p=src;\
	do {\
		while(SPI2STATbits.SPITBF);\
		SPI2BUF=*p++;\
		while(SPI2STATbits.SPITBF);\
		SPI2BUF=*p++;\
	}\
	while(--c);}
*****/

#define RCVR_SPI_MULTI(dst,cnt)	{\
	UINT c=cnt/2;\
	BYTE *p=dst;\
	do {\
		SPI2BUF=0xFF;\
		while(!SPI2STATbits.SPIRBF);\
		*p++=SPI2BUF;\
		SPI2BUF=0xFF;\
		while(!SPI2STATbits.SPIRBF);\
		*p++=SPI2BUF;\
	} while(--c);}


#endif  // !USE_DMA

/*-----------------------------------------------------------------------*/
/* Wait for card ready                                                   */
/*-----------------------------------------------------------------------*/

static
int wait_ready (void)
{
	BYTE d;

	Timer2 = 500;	/* Wait for ready in timeout of 500ms */
	do {
		d = xchg_spi(0xFF);
	} while ((d != 0xFF) && Timer2);

	return (d == 0xFF) ? 1 : 0;
}



/*-----------------------------------------------------------------------*/
/* Deselect the card and release SPI bus                                 */
/*-----------------------------------------------------------------------*/

static
void deselect (void)
{
	CS_HIGH();
	xchg_spi(0xFF);		/* Dummy clock (force DO hi-z for multiple slave SPI) */
}



/*-----------------------------------------------------------------------*/
/* Select the card and wait ready                                        */
/*-----------------------------------------------------------------------*/

static
int select (void)	/* 1:Successful, 0:Timeout */
{
	CS_LOW();
	xchg_spi(0xFF);		/* Dummy clock (force DO enabled) */

	if (wait_ready()) return 1;	/* OK */
	deselect();
	return 0;	/* Timeout */
}

const BYTE dummytxbuf[]={
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
};


/*-----------------------------------------------------------------------*/
/* Receive a data packet from MMC                                        */
/*-----------------------------------------------------------------------*/

static
int rcvr_datablock (	/* 1:OK, 0:Failed */
	BYTE *buff,			/* Data buffer to store received data */
	UINT btr			/* Byte count (must be multiple of 4) */
)
{
	BYTE token;

	Timer1 = 100;
	do {							/* Wait for data packet in timeout of 100ms */
		token = xchg_spi(0xFF);
	} while ((token == 0xFF) && Timer1);

	if(token != 0xFE) return 0;		/* If not valid data token, retutn with error */

#if USE_DMA
	DmaChnSetTxfer(SPI_TX_DMA_CH, &dummytxbuf, (void*)&SPI2BUF, btr, 1,1);
    DmaChnSetTxfer(SPI_RX_DMA_CH, (void*)&SPI2BUF, buff,1,btr, 1);
    DmaChnEnable(SPI_RX_DMA_CH);
    DmaChnEnable(SPI_TX_DMA_CH);
	tsk_chg(BGTskID);	// run background task
#else
	RCVR_SPI_MULTI(buff, btr);		/* Receive the data block into buffer */
#endif
	xchg_spi(0xFF);					/* Discard CRC */
	xchg_spi(0xFF);

	return 1;						/* Return with success */
}



/*-----------------------------------------------------------------------*/
/* Send a data packet to MMC                                             */
/*-----------------------------------------------------------------------*/

#if _USE_WRITE
static
int xmit_datablock (	/* 1:OK, 0:Failed */
	const BYTE *buff,	/* 512 byte data block to be transmitted */
	BYTE token			/* Data token */
)
{
	BYTE resp;

	if (!wait_ready()){
		return 0;
	}
	xchg_spi(token);		/* Xmit a token */

	if (token != 0xFD) {	/* Not StopTran token */
#if USE_DMA
	    DmaChnSetTxfer(SPI_TX_DMA_CH, buff, (void*)&SPI2BUF, 512, 1, 1);
	    DmaChnEnable(SPI_TX_DMA_CH);
		tsk_chg(BGTskID); // run background task
#else
		XMIT_SPI_MULTI(buff, 512);	/* Xmit the data block to the MMC */
#endif
		while(SPI2STATbits.SPIBUSY);
		SPI2STATbits.SPIROV=0;
		xchg_spi(0xFF);				/* CRC (Dummy) */
		xchg_spi(0xFF);
		resp = xchg_spi(0xFF);		/* Receive a data response */
		if ((resp & 0x1F) != 0x05){	/* If not accepted, return with error */
			return 0;
		}
	}
	return 1;
}
#endif


#if	USE_DMA
/*-----------------------------------------------------------------------*/
/* DMA ch0 interrupt routine                                             */
/*-----------------------------------------------------------------------*/

void __attribute__( (interrupt(ipl2), vector(_DMA_0_VECTOR))) _DMA0InterruptWrapper( void );
void _DMA0Interrupt(void)
{
    int    evFlags;      // event flags when getting the interrupt

    mDmaChnClrIntFlag(0); // release the interrupt in the INT controller, we're servicing int
    evFlags=DmaChnGetEvFlags(0); // get the event flags

    if(evFlags&DMA_EV_BLOCK_DONE)
    { // just a sanity check. we enabled just the DMA_EV_BLOCK_DONE transfer done interrupt
		DmaChnClrEvFlags(0, DMA_EV_BLOCK_DONE);
    }
}


/*-----------------------------------------------------------------------*/
/* DMA ch1 interrupt routine                                             */
/*-----------------------------------------------------------------------*/

void __attribute__( (interrupt(ipl2), vector(_DMA_1_VECTOR))) _DMA1InterruptWrapper( void );
void _DMA1Interrupt(void)
{
    int    evFlags;                // event flags when getting the interrupt

    mDmaChnClrIntFlag(1);        // release the interrupt in the INT controller, we're servicing int
    evFlags=DmaChnGetEvFlags(1);    // get the event flags

    if(evFlags&DMA_EV_BLOCK_DONE)
    { // just a sanity check. we enabled just the DMA_EV_BLOCK_DONE transfer done interrupt
		DmaChnClrEvFlags(1, DMA_EV_BLOCK_DONE);
    }
}
#else
	void _DMA0Interrupt(void){;}   // reference from dma0_isr.S
	void _DMA1Interrupt(void){;}
#endif


/*-----------------------------------------------------------------------*/
/* Send a command packet to MMC                                          */
/*-----------------------------------------------------------------------*/

static
BYTE send_cmd (
	BYTE cmd,		/* Command byte */
	DWORD arg		/* Argument */
)
{
	BYTE n, res;

	if (cmd & 0x80) {	/* ACMD<n> is the command sequense of CMD55-CMD<n> */
		cmd &= 0x7F;
		res = send_cmd(CMD55, 0);
		if (res > 1) return res;
	}

	/* Select the card and wait for ready */
	deselect();
	if (!select()) return 0xFF;

	/* Send command packet */
	xchg_spi(0x40 | cmd);			/* Start + Command index */
	xchg_spi((BYTE)(arg >> 24));	/* Argument[31..24] */
	xchg_spi((BYTE)(arg >> 16));	/* Argument[23..16] */
	xchg_spi((BYTE)(arg >> 8));		/* Argument[15..8] */
	xchg_spi((BYTE)arg);			/* Argument[7..0] */
	n = 0x01;						/* Dummy CRC + Stop */
	if (cmd == CMD0) n = 0x95;		/* Valid CRC for CMD0(0) + Stop */
	if (cmd == CMD8) n = 0x87;		/* Valid CRC for CMD8(0x1AA) + Stop */
	xchg_spi(n);

	/* Receive command response */
	if (cmd == CMD12) xchg_spi(0xFF);	/* Skip a stuff byte on stop to read */
	n = 10;							/* Wait for a valid response in timeout of 10 attempts */
	do
		res = xchg_spi(0xFF);
	while ((res & 0x80) && --n);

	return res;			/* Return with the response value */
}



/*--------------------------------------------------------------------------

   Public Functions

---------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------*/
/* Initialize Disk Drive                                                 */
/*-----------------------------------------------------------------------*/

DSTATUS disk_initialize (
	BYTE drv		/* Physical drive number (0) */
)
{
	BYTE n, cmd, ty, ocr[4];
	BYTE highspeed=0;

	if (drv) return STA_NOINIT;			/* Supports only single drive */
	if (Stat & STA_NODISK) return Stat;	/* No card in the socket */

	power_on();							/* Force socket power on */
	FCLK_SLOW();
	Stat |= STA_NOINIT;	// S.Suwa


	for (n = 10; n; n--) xchg_spi(0xFF);	/* 80 dummy clocks */

	ty = 0;
	if (send_cmd(CMD0, 0) == 1) {			/* Enter Idle state */
		Timer1 = 1000;						/* Initialization timeout of 1000 msec */

		if (send_cmd(CMD8, 0x1AA) == 1) {	/* SDv2? */
			for (n = 0; n < 4; n++) ocr[n] = xchg_spi(0xFF);			/* Get trailing return value of R7 resp */
			if (ocr[2] == 0x01 && ocr[3] == 0xAA) {				/* The card can work at vdd range of 2.7-3.6V */
				while (Timer1 && send_cmd(ACMD41, 0x40000000));	/* Wait for leaving idle state (ACMD41 with HCS bit) */
				if (Timer1 && send_cmd(CMD58, 0) == 0) {			/* Check CCS bit in the OCR */
					for (n = 0; n < 4; n++) ocr[n] = xchg_spi(0xFF);
					ty = (ocr[0] & 0x40) ? CT_SD2|CT_BLOCK : CT_SD2;	/* SDv2 */
				}
			}

if(send_cmd(CMD6,0x00ffff01)==0){
	for(n=0;n<64;n++){
		xchg_spi(0xFF);	//dummy read status
	}
	if(send_cmd(CMD6,0x80ffff01)==0){
		highspeed=1;
		for(n=0;n<64;n++){
			xchg_spi(0xFF);	//dummy read status
		}
		//switch to high-speed mode
	}
}

		} else {							/* SDv1 or MMCv3 */
			if (send_cmd(ACMD41, 0) <= 1) 	{
				ty = CT_SD1; cmd = ACMD41;	/* SDv1 */
			} else {
				ty = CT_MMC; cmd = CMD1;	/* MMCv3 */
			}
			while (Timer1 && send_cmd(cmd, 0));		/* Wait for leaving idle state */
			if (!Timer1 || send_cmd(CMD16, 512) != 0)	/* Set read/write block length to 512 */
				ty = 0;
		}
	}
	CardType = ty;
	deselect();

	if (ty) {			/* Initialization succeded */
		Stat &= ~STA_NOINIT;	/* Clear STA_NOINIT */
		if(highspeed){
			FCLK_FAST();		//sibu over 25MHz
		}else{
			FCLK_MID();			//sibu max 25MHz
		}
	} else {			/* Initialization failed */
		power_off();
	}

	return Stat;

}



/*-----------------------------------------------------------------------*/
/* Get Disk Status                                                       */
/*-----------------------------------------------------------------------*/

DSTATUS disk_status (
	BYTE pdrv		/* Physical drive nmuber (0) */
)
{
	if (pdrv) return STA_NOINIT;	/* Supports only single drive */
	return Stat;
}



/*-----------------------------------------------------------------------*/
/* Read Sector(s)                                                        */
/*-----------------------------------------------------------------------*/

DRESULT disk_read (
	BYTE pdrv,		/* Physical drive nmuber (0) */
	BYTE *buff,		/* Pointer to the data buffer to store read data */
	DWORD sector,	/* Start sector number (LBA) */
	BYTE count		/* Sector count (1..255) */
)
{
	if (pdrv || !count) return RES_PARERR;
	if (Stat & STA_NOINIT) return RES_NOTRDY;

	if (!(CardType & CT_BLOCK)) sector *= 512;	/* Convert to byte address if needed */

	if (count == 1) {		/* Single block read */
		if ((send_cmd(CMD17, sector) == 0)	/* READ_SINGLE_BLOCK */
			&& rcvr_datablock(buff, 512))
			count = 0;
	}
	else {				/* Multiple block read */
		if (send_cmd(CMD18, sector) == 0) {	/* READ_MULTIPLE_BLOCK */
			do {
				if (!rcvr_datablock(buff, 512)) break;
				buff += 512;
			} while (--count);
			send_cmd(CMD12, 0);				/* STOP_TRANSMISSION */
		}
	}
	deselect();

	return count ? RES_ERROR : RES_OK;
}



/*-----------------------------------------------------------------------*/
/* Write Sector(s)                                                       */
/*-----------------------------------------------------------------------*/

#if _USE_WRITE
DRESULT disk_write (
	BYTE pdrv,				/* Physical drive nmuber (0) */
	const BYTE *buff,		/* Pointer to the data to be written */
	DWORD sector,			/* Start sector number (LBA) */
	BYTE count				/* Sector count (1..255) */
)
{
	if (pdrv || !count) return RES_PARERR;
	if (Stat & STA_NOINIT) return RES_NOTRDY;
	if (Stat & STA_PROTECT) return RES_WRPRT;

	if (!(CardType & CT_BLOCK)) sector *= 512;	/* Convert to byte address if needed */

	if (count == 1) {		/* Single block write */
		if (send_cmd(CMD24, sector) == 0){	/* WRITE_BLOCK */
			if(xmit_datablock(buff, 0xFE)){
				count = 0;
			}
		}
	}else {				/* Multiple block write */
		if (CardType & CT_SDC) send_cmd(ACMD23, count);
		if (send_cmd(CMD25, sector) == 0) {	/* WRITE_MULTIPLE_BLOCK */
			do {
				if (!xmit_datablock(buff, 0xFC)) break;
				buff += 512;
			} while (--count);
			if (!xmit_datablock(0, 0xFD))	/* STOP_TRAN token */
				count = 1;
		}
	}
	deselect();

	return count ? RES_ERROR : RES_OK;
}
#endif



/*-----------------------------------------------------------------------*/
/* Miscellaneous Functions                                               */
/*-----------------------------------------------------------------------*/

#if _USE_IOCTL
DRESULT disk_ioctl (
	BYTE pdrv,		/* Physical drive nmuber (0) */
	BYTE cmd,		/* Control code */
	void *buff		/* Buffer to send/receive data block */
)
{
	DRESULT res;
	BYTE n, csd[16], *ptr = buff;
	DWORD csz;

	if (pdrv) return RES_PARERR;
	if (Stat & STA_NOINIT) return RES_NOTRDY;

	res = RES_ERROR;
	switch (cmd) {
	case CTRL_SYNC :	/* Flush write-back cache, Wait for end of internal process */
		if (select()){
		//	deselect();
			res = RES_OK;
		}
		break;

	case GET_SECTOR_COUNT :	/* Get number of sectors on the disk (WORD) */
		if ((send_cmd(CMD9, 0) == 0) && rcvr_datablock(csd, 16)) {
			if ((csd[0] >> 6) == 1) {	/* SDv2? */
				csz = csd[9] + ((WORD)csd[8] << 8) + ((DWORD)(csd[7] & 63) << 16) + 1;
				*(DWORD*)buff = csz << 10;
			} else {					/* SDv1 or MMCv3 */
				n = (csd[5] & 15) + ((csd[10] & 128) >> 7) + ((csd[9] & 3) << 1) + 2;
				csz = (csd[8] >> 6) + ((WORD)csd[7] << 2) + ((WORD)(csd[6] & 3) << 10) + 1;
				*(DWORD*)buff = csz << (n - 9);
			}
			res = RES_OK;
		}else{
		}
		break;

//	case GET_SECTOR_SIZE :	/* Get sectors on the disk (WORD) */
//		*(WORD*)buff = 512;
//		res = RES_OK;
//		break;

	case GET_BLOCK_SIZE :	/* Get erase block size in unit of sectors (DWORD) */
		if (CardType & CT_SD2) {	/* SDv2? */
			if (send_cmd(ACMD13, 0) == 0) {		/* Read SD status */
				xchg_spi(0xFF);
				if (rcvr_datablock(csd, 16)) {				/* Read partial block */
					for (n = 64 - 16; n; n--) xchg_spi(0xFF);	/* Purge trailing data */
					*(DWORD*)buff = 16UL << (csd[10] >> 4);
					res = RES_OK;
				}
			}
		} else {					/* SDv1 or MMCv3 */
			if ((send_cmd(CMD9, 0) == 0) && rcvr_datablock(csd, 16)) {	/* Read CSD */
				if (CardType & CT_SD1) {	/* SDv1 */
					*(DWORD*)buff = (((csd[10] & 63) << 1) + ((WORD)(csd[11] & 128) >> 7) + 1) << ((csd[13] >> 6) - 1);
				} else {					/* MMCv3 */
					*(DWORD*)buff = ((WORD)((csd[10] & 124) >> 2) + 1) * (((csd[11] & 3) << 3) + ((csd[11] & 224) >> 5) + 1);
				}
				res = RES_OK;
			}
		}
		break;

	case MMC_GET_TYPE :		/* Get card type flags (1 byte) */
		*ptr = CardType;
		res = RES_OK;
		break;

	case MMC_GET_CSD :	/* Receive CSD as a data block (16 bytes) */
		if ((send_cmd(CMD9, 0) == 0)	/* READ_CSD */
			&& rcvr_datablock(buff, 16))
			res = RES_OK;
		break;

	case MMC_GET_CID :	/* Receive CID as a data block (16 bytes) */
		if ((send_cmd(CMD10, 0) == 0)	/* READ_CID */
			&& rcvr_datablock(buff, 16))
			res = RES_OK;
		break;

	case MMC_GET_OCR :	/* Receive OCR as an R3 resp (4 bytes) */
		if (send_cmd(CMD58, 0) == 0) {	/* READ_OCR */
			for (n = 0; n < 4; n++)
				*((BYTE*)buff+n) = xchg_spi(0xFF);
			res = RES_OK;
		}
		break;

	case MMC_GET_SDSTAT :	/* Receive SD statsu as a data block (64 bytes) */
		if ((CardType & CT_SD2) && send_cmd(ACMD13, 0) == 0) {	/* SD_STATUS */
			xchg_spi(0xFF);
			if (rcvr_datablock(buff, 64))
				res = RES_OK;
		}
		break;

	default:
		res = RES_PARERR;
	}

	deselect();

	return res;
}
#endif


/*-----------------------------------------------------------------------*/
/* Device Timer Driven Procedure                                         */
/*-----------------------------------------------------------------------*/
/* This function must be called by timer interrupt in period of 1ms      */

void disk_timerproc (void)
{
	BYTE s;
	UINT n;


	n = Timer1;					/* 1000Hz decrement timer with zero stopped */
	if (n) Timer1 = --n;
	n = Timer2;
	if (n) Timer2 = --n;


	/* Update socket status */

	s = Stat;

	if (WP) s |= STA_PROTECT;
	else		s &= ~STA_PROTECT;

	if (INS) s &= ~STA_NODISK;
	else		 s |= (STA_NODISK | STA_NOINIT);

	Stat = s;
}

/*** end of mmc.c *********************************************/
