/******************************************************************************

    main.c

	Drive Recorder by REPIC32SD board

	2013/7/16  S.Suwa http://www.suwa-koubou.jp	

	PIC           microchip PIC32MX695F512H
	WebCamera     logicool  C525
	Memory disk   8GB SDCard
	Display       epson SD13781 QVGA(320x240)

       assign

       EPSON LCDC SD13781 INDIRECT 8 bit mode INTERFACE

       REPIC32SD Board   LCDC SD13781
        pin number         pin number
	    RB0(P16) J3-4  --- J4-27 CS#_SCS#
	    RB1(P15) J3-5  --- J4-25 RD#
	    RB2(P14) J3-7  --- J4-28 WR#_SCK
	    RB3(P13) J3-6  --- J4-6  P/C#_AB1

	    RD0(P46) J4-6  --- J4-29 DB0_SI
	    RD1(P49) J4-7  --- J4-30 DB1_SO
	    RD2(P50) J4-8  --- J4-31 DB2
	    RD3(P51) J4-9  --- J4-32 DB3
	    RD4(P52) J4-10 --- J4-33 DB4
	    RD5(P53) J4-11 --- J4-34 DB5
 	    RD6(P54) J4-12 --- J4-35 DB6
 	    RD7(P55) J4-13 --- J4-36 DB7

      UART2
	    U2TX(P32) J4-1  --- U2TX
	    U2RX(P31) J4-2  --- U2RX

      Monitor
        RB12           --- LED1 internal connection
        RB13           --- LED2 internal connection
        RB15(P30)J3-15 --- TP test measurement terminal 

      MMC
        RG6/SCK2       --- MMC_SCK internal connection
        RG7/SDI2       --- MMC_SDO internal connection
        RG8/SDO2       --- MMC_SD1 internal connection
        RG9            --- MMC_CS  internal connection
        RB5(PU)        --- MMC_CD  internal connection
        RB4(PU)        --- MMC_WP  internal connection
        RC14(PU)       --- MMC_DATA2 internal connection
        RC13(PU)       --- MMC_DATA1 internal connection

      Switch
        RD9 (P43) J4-3  --- Sw1 (SET/PAUSE key)
        RD10(P44) J4-4  --- Sw2 (UP/STOP key)
        RD11(P45) J4-5  --- Sw3 (DOWN/STOP key)

    INTERRUPT PRIORITY LEVEL
        ipl2   _T1Interrupt    1msec interval timer
        ipl2   _DMA0InterruptWrapper
        ipl2   _DMA1InterruptWrapper
        ipl2   _T4InterruptWrapper
        ipl4   _USBInterrupt  USB event(DETACH,SOF,TRN,etc)

    ICSP(PICkit3)
        NMCLR      J3-1
        +3.3V      J3-2
        GND        J3-3
        PGED2(P18) J3-9    
        PGEC2(P17) J3-8
        

******************************************************************************/

#include "GenericTypeDefs.h"
#include "HardwareProfile.h"
#include "usb.h"
#include "usb_host_uvc.h"
#include "s1d13781.h"
#include "rtc.h"
#include "diskio.h"
#include "image.h"
#include "key.h"


/*------------------------------------------------------------------------------
	Configuration for PIC32MX695F 
------------------------------------------------------------------------------*/

#include "p32mx695f_config.h"


/*------------------------------------------------------------------------------
	definition
------------------------------------------------------------------------------*/

typedef enum {
	OP_SET_TIME=0,
    OP_MONITOR,
	OP_SD_RECORD,
	OP_SD_PLAY,
	OP_PHOTO_FRAME,
	OP_EXIT,
} OPCODE;

#define	putstr(x)	UART2PrintString(x)
#define	puthex(x)	UART2PutHex(x)
#define	putch(x)	UART2PutChar(x)

/*------------------------------------------------------------------------------
	function prototype
------------------------------------------------------------------------------*/

void   do_operation(void);
OPCODE select_menu(void);
void   display_device_state(void);
void   SetDateTime(void);
void   display_date_time(void);
BYTE   select_show_time(void);
BYTE   oprPause(void);
void   InitializeSystem(void);
BOOL   USB_ApplicationEventHandler(BYTE address, USB_EVENT event, void *data, DWORD size);


/*------------------------------------------------------------------------------
	global variable
------------------------------------------------------------------------------*/

// camera state
CAMERA_STATE camera_state = CAMERA_DETACHED;

/*==============================================================================
	main
===============================================================================*/

int main ( void )
{
    // I/O initialize 
    InitializeSystem();
	// UART2 initialize
	UART2Init();
	// RTC initialize
	rtc_init();
	// USB Host driver
	USBHostInit(0);
	// LCDC initialize
	lcdc_initialize();
	// 

	// task changer initialize
	tsk_init();
	tsk_entry(idle_task); // Task ID of idle task is 1.

	// Display title picture on LCD
	title_picture_on();

	putstr("Drive Recorder start.\r\n");

	// wait for Camera Attached during 1 sec
	AplTimer = 1000;
	while(camera_state == CAMERA_DETACHED && AplTimer){
		USBHostTasks();
	}

	// main operation loop
	for(;;){

		lcd_clear(PIP_LAYER, cBLACK);
		set_pip_transparency(tON);
		set_pip_blinking(tON);

		// wait for Device ready
		while(camera_state == CAMERA_DETACHED || (disk_status(0) & (STA_NODISK|STA_PROTECT))){
			USBHostTasks();
			display_device_state();
		}

		// wait for Camera Ready 
		delay_ms(500);

		// do operation
		do_operation();
	}
}


/*---------------------------------------------------------------------------------------------
	display device status on PIP_LAYER
---------------------------------------------------------------------------------------------*/

void display_device_state(void)
{
	LOCATE	x, y;

	// set display position
	x = 15; y = 8;

	// display camera status
	lcd_putstr_wide(PIP_LAYER, x, y, cRED, cBLACK, (camera_state == CAMERA_DETACHED)? "No Camera": "         ");
	y+=3;
	// display disk status
	lcd_putstr_wide(PIP_LAYER, x, y, cRED, cBLACK, (disk_status(0) & STA_NODISK)? "No Card": "       ");
	y+=3;
	lcd_putstr_wide(PIP_LAYER, x, y, cRED, cBLACK, (!(disk_status(0) & STA_NODISK))&&(disk_status(0) & STA_PROTECT)? "Card Protect": "            ");

} // display_device_state


/*---------------------------------------------------------------------------------------------
	do operations
---------------------------------------------------------------------------------------------*/

void do_operation(void)
{
	OPCODE opr;

	for(;;){

		// background display to start
		yuyv_display_start();

		opr = select_menu();

		switch(opr){
		case OP_SET_TIME:
			putstr("Set Date & Time.\r\n");
			SetDateTime();
			break;

		case OP_SD_PLAY:
			yuyv_display_stop();
			set_pip_transparency(tOFF);
			putstr("SD Play.\r\n");
			SD_Play();
			break;

		case OP_SD_RECORD:
			yuyv_display_stop();
			putstr("SD Record.\r\n");
			SD_Record();
			break;

		case OP_PHOTO_FRAME:
			if(select_show_time()){
				yuyv_display_stop();
				set_pip_transparency(tOFF);
				putstr("PhotoFrame.\r\n");
				PhotoFrame();
			}
			break;

		case OP_MONITOR:
			lcd_clear(PIP_LAYER, cBLACK);
			// wait keyin
			putstr("Video Monitor.\r\n");
			while(getkey() == KEY_NONE){
				USBHostTasks();
				if(camera_state == CAMERA_ATTACHED){
					select_display_layer(PIP_LAYER);
					yuyv_display(MAIN_LAYER);
					select_display_layer(MAIN_LAYER);
					yuyv_display(PIP_LAYER);
				}else{
					break;
				}
			}
			break;

		case OP_EXIT:
			// found device error
			putstr("Device error!!\r\n");
			// display stop!
			yuyv_display_stop();
			return;

		default:
			break;
		} // switch

	} // for

}


/*---------------------------------------------------------------------------------------------
	select operations menu
---------------------------------------------------------------------------------------------*/

static
const BYTE  *opstr[] = {
	" Set Date & Time ",
    " Video Monitor   ",
	" SD Record       ",
	" SD Play         ",
	" Photo Frame     ",
};

OPCODE	select_menu(void)
{
	KEYCODE	key;
	int    op, n;
	LOCATE x, y;
	WORD   fg, bg;

	// set pip layer transparency
	lcd_clear(PIP_LAYER, cBLACK);
	set_pip_transparency(tON);
	select_display_layer(PIP_LAYER);
	
	x = 10;	y = 6;

	for(op = 0; camera_state == CAMERA_ATTACHED && !(disk_status(0) & (STA_NODISK|STA_PROTECT)); ){

		// display operation menu
		for(n = 0; n < sizeof(opstr)/sizeof(opstr[0]);n++){
			fg = cRED;
			bg = (n == op)? cCYAN: cBLACK;
			lcd_putstr_wide(PIP_LAYER, x, y+(n*3), fg, bg, (BYTE *)opstr[n]);
		}

		// wait keyin
		while((key = getkey()) == KEY_NONE){

			// display Date and Time on LCD
			display_date_time();

			// call USB Task
			USBHostTasks();

			// display image QQVGA yuyv format 
			if(camera_state == CAMERA_ATTACHED){
				yuyv_display(MAIN_LAYER);
			}

			// device error?
			if(camera_state == CAMERA_DETACHED ||
				 (disk_status(0) & (STA_NODISK|STA_PROTECT))){
				return OP_EXIT;
			}
		}

		switch(key){
		case KEY_SET:
			return (OPCODE)op;

		case KEY_UP:
			if(--op < 0) op = (int)OP_PHOTO_FRAME;
			break;

		case KEY_DOWN:
			if(++op > (int)OP_PHOTO_FRAME) op = 0;
			break;
		}
	}

	return OP_EXIT;

}


/*---------------------------------------------------------------------------------------------
	select and set frame show time
---------------------------------------------------------------------------------------------*/

static
const struct {
	BYTE *str;
	DWORD val;
} tmval[] = {
    { " 10 sec ", 10000l, },
	{ " 30 sec ", 30000l, },
    { " 60 sec ", 60000l, },
    { "  3 min ", 180000l, },
	{ "  5 min ", 300000l, },
    { " 10 min ", 600000l, },
    { " Cancel ", 0l, },
};

BYTE select_show_time(void)
{
	KEYCODE	key;
	int    op, n;
	LOCATE x, y;
	WORD   fg, bg;

	// set pip layer transparency
	lcd_clear(PIP_LAYER, cBLACK);
	set_pip_transparency(tON);
	select_display_layer(PIP_LAYER);
	lcd_putstr_wide(PIP_LAYER, 10, 4, cBLUE, cYELLOW, " << Show Time >> ");

	x = 18;	y = 7;

	for(op = 0; camera_state == CAMERA_ATTACHED && !(disk_status(0) & (STA_NODISK|STA_PROTECT)); ){

		// display operation menu
		for(n = 0; n < sizeof(tmval)/sizeof(tmval[0]);n++){
			fg = cRED;
			bg = (n == op)? cCYAN: cBLACK;
			lcd_putstr_wide(PIP_LAYER, x, y+(n*2), fg, bg, tmval[n].str);
		}

		// wait keyin
		while((key = getkey()) == KEY_NONE){

			// call USB Task
			USBHostTasks();

			// display image QQVGA yuyv format 
			if(camera_state == CAMERA_ATTACHED){
				yuyv_display(MAIN_LAYER);
			}

			// device error?
			if(camera_state == CAMERA_DETACHED ||
				 (disk_status(0) & (STA_NODISK|STA_PROTECT))){
				return 0;	// return code '0' is "cancel".
			}
		}

		switch(key){
		case KEY_SET:
			if(tmval[op].val == 0) return 0;	// "cancel"
			set_show_time(tmval[op].val);
			return 1;	// set ok.

		case KEY_UP:
			if(--op < 0) op = sizeof(tmval)/sizeof(tmval[0]) -1;
			break;

		case KEY_DOWN:
			if(++op >= sizeof(tmval)/sizeof(tmval[0])) op = 0;
			break;
		}
	}

	return 0;	// return code is "cancel"

}


/*======================================================================================
	Set Date and Time

    tE̐ݒ@
	 NAAAjAAAb@̏ɐݒB
	@NJnASET@Ŏ̍ڂ֐iށB
	@b̐ݒ̎@SET@ŏI
	 UP,DOWNŊeڂ̒l+1,-1oB
=======================================================================================*/

// time set operation control data
const struct {
	int begin;
	int length;
	int max;
	int min;
} item[] = {
	{0, 4, 99, 0,}, // YYYY
	{5, 2, 12, 1,},	// MM
	{8, 2, 31, 1,},	// DD
	{11,2, 23, 0,}, // hh
	{14,2, 59, 0,}, // mm
	{17,2, 59, 0,}, // ss
};

void SetDateTime(void)
{
	char 	buf[20];	// YYYY/MM/DD hh:mm:ss length=19
	char	wbuf[3][20];
	char	data[6];
	int		i, key;
	LOCATE	x, y;

	// read Date & Time
	get_rtc(data);

	// set display position
	x = 6; y = 10;
	lcd_clear(PIP_LAYER, cBLACK);

	// Set from YYYY to ss (YYYY/MM/DD hh:mm:ss)
	for(i = 0; i < 6; ){

		// display Date & Time strings
		//  selected item is highlited
		sprintf(buf, "%4d/%02d/%02d %02d:%02d:%02d",
			data[0]+2000, data[1], data[2], data[3], data[4], data[5]);

		strncpy(wbuf[0], buf, item[i].begin)[item[i].begin] = 0;
		strncpy(wbuf[1], buf + item[i].begin, item[i].length)[item[i].length] = 0;
		strcpy (wbuf[2], buf + item[i].begin + item[i].length); 

		// display value
		lcd_putstr_wide(PIP_LAYER, x, y, cRED, cBLACK, wbuf[0]);
		lcd_putstr_wide(PIP_LAYER, x + item[i].begin *2, y, cBLUE, cYELLOW, wbuf[1]);
		lcd_putstr_wide(PIP_LAYER, x + (item[i].begin + item[i].length) *2, y, cRED, cBLACK, wbuf[2]);


		// wait for key 
		while((key = getkey()) == KEY_NONE);

		// change value
		if(key == KEY_UP){
			if(++data[i] > item[i].max) data[i] = item[i].min;
		}else if(key == KEY_DOWN){
			if(--data[i] < item[i].min) data[i] = item[i].max;
		}else if(key == KEY_SET){
			++i;
		}	
	}

	// set Now Date & Time
	rtc_set(buf);

}


/*======================================================================================
	display date & time on LCD
======================================================================================*/

volatile int one_sec_flg;	
// one_sec_flg set to 1 every second by T1 isr. See rtc.c

void	display_date_time(void)
{
	char	buf[20];

	if(!one_sec_flg) return;
	one_sec_flg = 0;
	get_rtcstr(buf);
	lcd_putstr(PIP_LAYER, 33, 0, cYELLOW, cBLACK, buf);
}


/*======================================================================================
	Pausing image operation
	When SET key was pushed, return "replay".
       other keys were pushed, return "stop".
	return code: replay = 1, stop = 0
=======================================================================================*/

BYTE	oprPause(void)
{
	LOCATE  x, y;
	KEYCODE key;

	// set pip layer transparency
	lcd_clear(PIP_LAYER, cBLACK);
	set_pip_transparency(tON);
	set_pip_blinking(tON);

	x = 24; y = 13;

	// display pause message
	lcd_putstr_wide(PIP_LAYER, x, y, cRED, cBLACK, "Pause");

	while((key = getkey()) == KEY_NONE){

		// call USB Host process
		USBHostTasks();

		// device error?
		if(camera_state == CAMERA_DETACHED ||
			 (disk_status(0) & (STA_NODISK|STA_PROTECT))){
			return 0;
		}
	}

	lcd_putstr_wide(PIP_LAYER, x, y, cRED, cBLACK, "     ");

	if(key == KEY_SET)
		return 1;
	else
		return 0;


} // oprPause


/*=======================================================================================
	InitializeSystem() for REPIC32SD Board
=======================================================================================*/

void InitializeSystem ( void )
{
	int	value;

	// Set System Config
	//SYSTEMConfigPerformance(GetSystemClock());
	value = SYSTEMConfigWaitStatesAndPB( GetSystemClock() );

	// Enable the cache for the best performance	
	CheKseg0CacheOn();

	// enable multi vector
	INTEnableSystemMultiVectoredInt();

	value = OSCCON;
	while (!(value & 0x00000020)){
		value = OSCCON;    // Wait for PLL lock to stabilize	
	}

	INTEnableInterrupts();

	// ADC input off
	AD1PCFG = 0xffff;
	// disable JTAG
	DDPCONbits.JTAGEN = 0;

	//--- setting I/O for S1D13781
	// indirect 8bit interface control
	// LATB = 0b0111;	// P/C#=0,WR#=1,RD#=1,CS#=1
	LATB = 0x3ff7;	// P/C#=0,WR#=1,RD#=1,CS#=1, LED1,LED2 turn off
	TRISB = 0x0ff0;	// control signal out
			    // RB0-RB3 for LCDC control, as output
				// RB4(MMC WP), RB5(MMC CD) as input
				// RB12(LED1),RB13(LED2) as output
				// RB14,RB15(monitor)
	LATD = 0; TRISD = 0xff00;   // RD0-RD7 as output, RD8-RD15 as input.
                // RD0-RD7   LCDC controle data out/input port
				// RD8(INT1) USB Isoc. frame data receved done interrupt
                // RD9(KEY_SET), RD10(KEY_UP), RD11(KEY_DOWN) 
	//--- setting UART2 degital port
	TRISF = 0x0010;	   // RF4(U2RX) is input.
	LATFSET = 0x0020;  // RF5(U2TX) to level high.

	//--- setting SPI2 for MMC
	LATG=	0x0000;
	TRISG=	0xfcbf;		//RG6(SCK), RG8(SDO), RG9(CS) as output

	//--- internal pullup
	CNPUEbits.CNPUE0=1;	// MMC_DAT1 Pull-up
	CNPUEbits.CNPUE1=1;	// MMC_DAT2 Pull-up
	CNPUEbits.CNPUE6=1;	// MMC_WP Pull-up
	CNPUEbits.CNPUE7=1;	// MMC_CD Pull-up
	CNPUEbits.CNPUE9=1;	// SDI2 Pull-up, external 50k pullup in circuit

	// Timer4 ISR
	IEC0CLR = _IEC0_T4IE_MASK; // disable T4 interrupt
	IFS0CLR = _IFS0_T4IF_MASK; // clear interrupt flag
	IPC4bits.T4IP = 2;         // ipl is 2


} // InitializeSystem


/*=========================================================================================
BOOL USB_ApplicationEventHandler ( BYTE address, USB_EVENT event, void *data, DWORD size )
==========================================================================================*/

BOOL USB_ApplicationEventHandler ( BYTE address, USB_EVENT event, void *data, DWORD size )
{

    // Handle specific events.
    switch (event){
    case EVENT_UVC_ATTACH:
		camera_state = CAMERA_ATTACHED;
        //putstr("Camera attached.\r\n");
		break;

    case EVENT_UVC_DETACH:
    	camera_state = CAMERA_DETACHED;
        //putstr( "Camera detached.\r\n" );
		break;

    default:
       break;
   }
   return TRUE;

} // USB_ApplicationEventHandler


/*** end of main.c *************************************************************************/
