/******************************************************************************

	image.c

	2013/7/16  S.Suwa http://www.suwa-koubou.jp	
******************************************************************************/

#include <stdio.h>
#include "HardwareProfile.h"
#include "GenericTypedefs.h"
#include "usb.h"
#include "usb_host_uvc.h"
#include "s1d13781.h"
#include "ff.h"
#include "diskio.h"
#include "tjpgd.h"
#include "image.h"
#include "key.h"
#include "rtc.h"


// work parameter 
#define	WRKFRAMES       2
#define	WRKLINES        32

// capture image buffer size
#define	FRMBFSZ		( QQVGA_SIZE * sizeof(WORD) * WRKFRAMES ) // QQVGA, RGB565, 2frames (76,800bytes) 
#define	VRAMBFSZ	( QVGA_WIDTH * sizeof(WORD) * WRKLINES )  // QVGA, RGB565, 32lines  (20,480bytes)
#define	HEADER_LEN		12	// length of payload header

// frame data receiving buffer
static BYTE  frameBuffer[FRMBFSZ]  __attribute__ ((aligned(sizeof(WORD))));
static BYTE *frame_wp;		// receive data write pointer
static BYTE *frame_rp;		// frame data read pointer
static BYTE *frame_sp;		// frame start pointer
static int	 frame_skip;	// skip flag
static DWORD frame_size;	// frame size counter 
static DWORD frame_bytes;	// frameBuffer data counter

// queue of received frame notify 
#define	QUEUE_SIZE    4
#define QUEUE_MASK    3

static
struct {
    volatile WORD	frame_valid;
    volatile BYTE  *frame_sp;
    volatile DWORD	frame_size;
} frameQueue[QUEUE_SIZE];

static int	queuePushIndex, queuePopIndex;

// USB call back function pointer.
void	(*ISOC_READ_func)(BYTE *data, DWORD size);

// work buffer for
//  scaling up from qqvga to qvga display, and
//  making string(font) data for LCD display, and
//  display recoding frame data store.
BYTE   vramBuffer[VRAMBFSZ]  __attribute__ ((aligned(sizeof(WORD))));

static BYTE	*frame_bp;	// point to frameBuffer, or jpeg frame etc.
static DWORD vram_addr;	// LCD vram address
static BYTE	 vram_sw;	// LCD layer switch, Main or Pip

// tjpgd work buffer
static JDEC	  jdec;
static BYTE   tjpgd_pool[3000];

// QQVGA display control state
#define STAT_STOP	0
#define STAT_START	1
#define	STAT_PAUSE	2

static int yuyv_stat;

// Camera State
extern	CAMERA_STATE camera_state; // See. main.c


#define	putstr(x)	UART2PrintString(x)
#define	puthex(x)	UART2PutHex(x)
#define	putch(x)	UART2PutChar(x)

// *************************************************************************
// *************************************************************************
// Section:  display image to LCD 
// *************************************************************************
// *************************************************************************

/*========================================================================
	display title picture (jpeg format) 
========================================================================*/

#include "title.h"	// title picture image data

void	title_picture_on(void)
{
	// write to LCD MAIN LAYER
	vram_addr = MAIN_VRAM_ADR;

	frame_bp = (BYTE *)title_picture;
	if(jd_prepare(&jdec, image_in_func, tjpgd_pool, sizeof(tjpgd_pool), NULL) == JDR_OK){
		jd_decomp(&jdec, lcd_out_func, 0);
	}
}


/*========================================================================
	display QQVGA image data (yuyv format)  
========================================================================*/
//
// start display
//
BYTE yuyv_display_start(void)
{
	int	maxformat, n;

	switch(yuyv_stat){
	case STAT_STOP:
		// support YUYV format?
		maxformat = USBHostUVCGetFormatCount();
		for(n = 0; n < maxformat; n++){
			if(USBHostUVCGetFormat(n) == FMT_YUY2){
				break;
			}
		}
		if(n >= maxformat) return UVC_UNSUPPORTED_FORMAT;

		// support QQVGA ?
		if(USBHostUVCSetFrame(n, QQVGA_WIDTH, QQVGA_HEIGHT) == UVC_UNSUPPORTED_RESOLUTION){
			return UVC_UNSUPPORTED_RESOLUTION;
		}

		// reset frame buffer pointer
		frame_wp = frame_sp = frameBuffer;
		frame_bytes = frame_size = 0;
		frame_skip = 1;
		queuePushIndex = queuePopIndex = 0;
		for(n = 0; n < QUEUE_SIZE; n++) frameQueue[n].frame_valid = 0;

		// Isochronous read, set call back function.
		ISOC_READ_func = receive_frame_data;

		// capture start
		if(USBHostUVCStartFrameCapturing() == UVC_FAIL){
			ISOC_READ_func = NULL;
			return UVC_FAIL;
		}
		yuyv_stat = STAT_START;
		return UVC_SUCCESS;

	case STAT_START:
		return UVC_SUCCESS;

	case STAT_PAUSE:
		yuyv_stat = STAT_START;
		return UVC_SUCCESS;

	default:
		return UVC_INVALID;
	}
}

//
// pause display
//
BYTE yuyv_display_pause(void)
{
	if(yuyv_stat == STAT_START) yuyv_stat = STAT_PAUSE;
	return UVC_SUCCESS;
}

//
// stop display
//
BYTE yuyv_display_stop(void)
{
	switch(yuyv_stat){
	case STAT_STOP:
		return UVC_SUCCESS;

	case STAT_PAUSE:
	case STAT_START:
		// Stop Frame Capture
		USBHostUVCStopFrameCapturing();
		// disable call back
		ISOC_READ_func = NULL;

		yuyv_stat = STAT_STOP;
		return UVC_SUCCESS;
	}
}

//
// one frame display
//
void yuyv_display(BYTE layer)
{
	if(yuyv_stat != STAT_START) return;

	// set LCD address
	vram_addr = (layer == MAIN_LAYER ? MAIN_VRAM_ADR: PIP_VRAM_ADR);

	// wait for received one frame 
	while(frameQueue[queuePopIndex].frame_valid == 0){
		USBHostTasks();
		if(camera_state == CAMERA_DETACHED) break;
	}

	// display one frame
	qqvga_display((BYTE *)frameQueue[queuePopIndex].frame_sp);

	// update bytes in frameBuffer
	IEC1CLR = _IEC1_USBIE_MASK;
	frame_bytes -= frameQueue[queuePopIndex].frame_size;
	IEC1SET = _IEC1_USBIE_MASK;
	frameQueue[queuePopIndex].frame_valid = 0;
	queuePopIndex++;
	queuePopIndex &= QUEUE_MASK;
}


/*-------------------------------------------------------------------
	one frame display
	convert YUYV to RGB565, scale up from QQVGA to QVGA
-------------------------------------------------------------------*/

void qqvga_display(BYTE *frame)
{
	WORD *dst, *src, x, y;

	// convert YCrCb --> RGB565
	yuv2rgb_color(frame, frame, 160 * 120 * 2);
	src = (WORD *)frame;
	// scaling up from QQVGA to QVGA 
	for(y = 0; y < 119; y++){
		// odd line
		dst = (WORD *)vramBuffer;
		for(x = 0; x < 159; x++){
			dst[0] = src[0];
			dst[1] = avr_rgb(src[0], src[1]); 
			dst += 2;
			src++;
		}
		dst[0] = src[0];
		dst[1] = 0;	     // right edge to BLACK
		write_byte_block(vram_addr, vramBuffer, 320 * 2); 
		vram_addr += (320 * 2);

		// next even line
		src -= 159;
		dst = (WORD *)vramBuffer;
		for(x = 0; x < 159; x++){
			dst[0] = avr_rgb(src[0], src[0 + 160]);
			dst[1] = avr_rgb( avr_rgb(src[0], src[1]), avr_rgb(src[0 + 160], src[1 + 160]));
			dst += 2;
			src++;
		}
		dst[0] = avr_rgb(src[0], src[0 + 160]);
		dst[1] = 0;
		write_byte_block(vram_addr, vramBuffer, 320 * 2); 
		vram_addr += (320 * 2);

		src++; 
	}

	// bottom 2 lines
	dst = (WORD *)vramBuffer;
	for(x = 0; x < 159; x++){
		dst[0] = src[0];
		dst[1] = avr_rgb(src[0], src[1]); 
		dst += 2;
		src++;
	}
	dst[0] = src[0];
	dst[1] = 0;	     // right edge to BLACK
	write_byte_block(vram_addr, vramBuffer, 320 * 2); 
	vram_addr += (320 * 2);

	// next line
	dst = (WORD *)vramBuffer;
	for(x = 0; x < 160; x++){
		dst[0] = 0;
		dst[1] = 0;
		dst += 2;
	}
	write_byte_block(vram_addr, vramBuffer, 320 * 2); 
	vram_addr += (320 * 2);

}


/*-------------------------------------------------------------------
	convert from YUYV to RGB565
-------------------------------------------------------------------*/

void	yuv2rgb_color(BYTE *dst, BYTE *src, WORD length)
{
	int y, u, v, i, j, k, l;
	int	r, g, b;
	WORD n;

	// convert from YUYV to RGB565
	for(n = 0; n < length; n+=4, src+=4){

		u = (int)(*(src+1)) - 128;
		v = (int)(*(src+3)) - 128;

		// YUV to RGB24
		//  for 32 bits MICOM
		i = 359 * v;
		j = 88 * u;
		k = 183 * v;
		l = 454 * u;

		y = (int)(*src) << 8;
		r = (y + i) >> 8;
		g = (y - j - k) >> 8;
		b = (y + l) >> 8;

		if(r < 0) r = 0;
		if(r > 255) r = 255;

		if(g < 0) g = 0;
		if(g > 255) g = 255;

		if(b < 0) b = 0;
		if(b > 255) b = 255;

		// RGB24 to RGB565
		*dst++ = ((g << 3) & 0b11100000) | ((b >> 3) & 0b00011111);	// low byte
		*dst++ = ((r & 0b11111000) | ((g >> 5) & 0b00000111)); // high byte

		// YUV to RGB24
		y = (int)(*(src+2)) << 8;
		r = (y + i) >> 8;
		g = (y - j - k) >> 8;
		b = (y + l) >> 8;

		if(r < 0) r = 0;
		if(r > 255) r = 255;

		if(g < 0) g = 0;
		if(g > 255) g = 255;

		if(b < 0) b = 0;
		if(b > 255) b = 255;

		// RGB24 to RGB565
		*dst++ = ((g << 3) & 0b11100000) | ((b >> 3) & 0b00011111);	// low byte
		*dst++ = ((r & 0b11111000) | ((g >> 5) & 0b00000111)); // high byte

	}
}

/*-------------------------------------------------------------------
	calcurate average RGB
-------------------------------------------------------------------*/

WORD avr_rgb(WORD rgb1, WORD rgb2)
{
	WORD r, g, b;

	r = (((rgb1 & 0xf800)>>1) + ((rgb1 & 0xf800)>>1)) & 0xf800;
	g = (((rgb1 & 0x07e0) + (rgb2 & 0x07e0)) >> 1) & 0x07e0;
	b = (((rgb1 & 0x001f) + (rgb2 & 0x001f)) >> 1) & 0x001f;

	return  r|g|b ;
}


// *************************************************************************
// *************************************************************************
// Section:  SD Card operation 
// *************************************************************************
// *************************************************************************

/*========================================================================
	SD play (read from CAPxxxxx.AVI)
========================================================================*/

static volatile WORD PLAY_SKIP_FRAMES;

BYTE SD_Play(void)
{
	FATFS	fs;
	FIL	    fin;
	UINT    cnt, fcnt;
	BYTE	c, flg, eof, *bp;
	WORD	skip;
	KEYCODE key;
	char    fname[13];
    FRESULT fres;

	f_mount(0, &fs);

	for(fcnt = 0; fcnt <= 99999; fcnt++){
		sprintf(fname, "CAP%05d.AVI", fcnt);
		if(f_open(&fin, fname, FA_READ) != FR_OK){
			break;
		}
		putstr(fname); putstr(" playing ...\r\n");

		// write to PIP
		select_display_layer(MAIN_LAYER);
		vram_addr = PIP_VRAM_ADR;
		vram_sw = PIP_LAYER;
	
		frame_bytes = 0;
		frame_rp = frameBuffer;
		eof = flg = 0;
		PLAY_SKIP_FRAMES = 6;

		for(;;){
			// check pause 
			if((key = getkey()) != KEY_NONE){
				switch(key){
				case KEY_SET:
					if(oprPause() == 0) goto error_exit;
					break;
				case KEY_UP:
					PLAY_SKIP_FRAMES += 2;
					break;
				case KEY_DOWN:
					PLAY_SKIP_FRAMES -= 2;
					if(PLAY_SKIP_FRAMES < 0) PLAY_SKIP_FRAMES = 1;
					break;
				}
			}

			LATBCLR=0x2000;
			for(skip = 0; skip < PLAY_SKIP_FRAMES; skip++){
				// read one frame 
				frame_sp = frame_rp;
				while(1){
					if(frame_bytes == 0){
						if(eof) goto error_exit;
						if(frame_rp - frame_sp) memcpy(frameBuffer, frame_sp, frame_rp - frame_sp);
						if(f_read(&fin, frameBuffer + (frame_rp - frame_sp), FRMBFSZ -(frame_rp - frame_sp), &cnt) != FR_OK) goto error_exit;
						if(cnt == 0) goto error_exit;
						if(cnt < (FRMBFSZ -(frame_rp - frame_sp))) eof = 1;
						frame_bytes = cnt;
						frame_rp = frameBuffer + (frame_rp - frame_sp);
						frame_sp = frameBuffer;
					}

					frame_bytes--;
					if(flg){
						flg = 0;
						if(*frame_rp++ == 0xd9){
							break;
						}
					}else{
						if(*frame_rp++ == 0xff){
							flg = 1;
						}
					} // if()
				} // while()
			} // for()			
			LATBSET=0x2000;
	
			// JPEG decord
			frame_bp = frame_sp;
			if(jd_prepare(&jdec, image_in_func, tjpgd_pool, sizeof(tjpgd_pool), NULL) == JDR_OK){
				jd_decomp(&jdec, lcd_out_func, 0);
			}

			// toggle display layer
			if(vram_sw == MAIN_LAYER){
				select_display_layer(MAIN_LAYER);
				vram_sw = PIP_LAYER;
				vram_addr = PIP_VRAM_ADR;
			}else{
				select_display_layer(PIP_LAYER);
				vram_sw = MAIN_LAYER;
				vram_addr = MAIN_VRAM_ADR;
			}
		}

error_exit:
		LATBSET=0x2000;
		putstr(" reached end of file.\r\n");
		f_close(&fin);
	} // for() to next file

	f_mount(0, NULL);
	return 0;
}


/*========================================================================
	SD record (write to CAPxxxxx.AVI)
========================================================================*/

#define	RECORD_SKIP_FRAMES  30
#define BGTskID              1
static int volatile display_complete;

BYTE    SD_Record(void)
{
	BYTE	maxformat, n;
	BYTE	*sp;
	DWORD	size;
	FATFS	fs;
	FIL	fout;
	UINT cnt;
	char    fname[13];
	FRESULT fres;

	// support mjpg format?
	maxformat = USBHostUVCGetFormatCount();
	for(n = 0; n < maxformat; n++){
		if(USBHostUVCGetFormat(n) == FMT_MJPEG){
			break;
		}
	}
	if(n >= maxformat){
		return UVC_UNSUPPORTED_FORMAT;
	}

	// support QVGA ?
	if(USBHostUVCSetFrame(n, QVGA_WIDTH, QVGA_HEIGHT) == UVC_UNSUPPORTED_RESOLUTION){
		return UVC_UNSUPPORTED_RESOLUTION;
	}

	// mount file system
	f_mount(0, &fs);

	// file open
	for(cnt=0; cnt <= 99999; cnt++){
		sprintf(fname, "CAP%05d.AVI", cnt);
		if((fres = f_open(&fout, fname, FA_CREATE_NEW | FA_WRITE)) == FR_OK){
			break;
		}else if(fres == FR_EXIST){
			continue;
		}else{
			f_mount(0, NULL);
			putstr(fname); putstr(" open error!!\r\n");
			return 1;
		}	
	}

	// reset frame buffer pointer
	frame_wp = frame_sp = frameBuffer;
	frame_bytes = frame_size = 0;
	frame_skip = RECORD_SKIP_FRAMES;
	queuePushIndex = queuePopIndex = 0;
	for(n = 0; n < 4; n++) frameQueue[n].frame_valid = 0;

	// Isochronous read, set call back function.
	ISOC_READ_func = receive_frame_data;

	// capture start
	if(USBHostUVCStartFrameCapturing() == UVC_FAIL){
		goto error_exit;
	}

	// write to PIP
	select_display_layer(MAIN_LAYER);
	vram_addr = PIP_VRAM_ADR;
	vram_sw = PIP_LAYER;
	set_pip_transparency(tOFF);

	// set background task
	tsk_delete(BGTskID);
	tsk_entry(display_recording_frame);
	display_complete = 1;
	IFS0CLR = _IFS0_T4IF_MASK;	// clear previous IF 
	IEC0SET = _IEC0_T4IE_MASK; // enable T4 interrupt

	// recording loop
	for(;camera_state == CAMERA_ATTACHED;){

		// check pause 
		if(getkey() != KEY_NONE){
			if(!oprPause()) break;
		}

		USBHostTasks();

		// wait for received one frame 
		while(frameQueue[queuePopIndex].frame_valid == 0){
			tsk_chg(BGTskID);
			if(camera_state == CAMERA_DETACHED) goto error_exit;
		}


		sp = (BYTE *)frameQueue[queuePopIndex].frame_sp;
		size = frameQueue[queuePopIndex].frame_size;

		// one frame SD card write
		LATBCLR=0x1000; // led1 on
		// ring over?
		if((sp + size) > (frameBuffer + FRMBFSZ)){
			if(f_write(&fout, sp, (frameBuffer + FRMBFSZ) - sp, &cnt) != FR_OK){
				goto error_exit;
			}		
			if(f_write(&fout, frameBuffer, (sp + size) - (frameBuffer + FRMBFSZ), &cnt) != FR_OK){
				goto error_exit;
			}		
		}else{
			if(display_complete && size <= VRAMBFSZ){
				memcpy(vramBuffer, sp, size);
				display_complete = 0;
			}
			if(f_write(&fout, sp, size, &cnt) != FR_OK){
				goto error_exit;
			}
		}
		LATBSET=0x1000;

		// update bytes
		IEC1CLR = _IEC1_USBIE_MASK;
		frame_bytes -= frameQueue[queuePopIndex].frame_size;
		IEC1SET = _IEC1_USBIE_MASK;
		frameQueue[queuePopIndex].frame_valid = 0;
		queuePopIndex++;
		queuePopIndex &= QUEUE_MASK;

		USBHostTasks();

	}

error_exit:
	LATBSET=0x1000;
	f_close(&fout);
	f_mount(0, NULL);

	// Stop Frame Capture
	USBHostUVCStopFrameCapturing();

	// disable call back
	ISOC_READ_func = NULL;

	// reset background task
	IEC0CLR = _IEC0_T4IE_MASK;
	tsk_delete(BGTskID);
	tsk_entry(idle_task);

	putstr(" stop recording.\r\n");
	return UVC_SUCCESS;
}


/*------------------------------------------------------------------------
	back ground display task
------------------------------------------------------------------------*/

void display_recording_frame(void)
{
	for(;;){

		while(display_complete){
			USBHostTasks();
			if(camera_state == CAMERA_DETACHED){
				tsk_chg(0);
			}
		}

		frame_bp = (BYTE *)vramBuffer;
		if(jd_prepare(&jdec, image_in_func, tjpgd_pool, sizeof(tjpgd_pool), NULL) == JDR_OK){
			jd_decomp(&jdec, lcd_out_func, 0);
		}

		// Time write and toggle display layer
		if(vram_sw == MAIN_LAYER){
			select_display_layer(MAIN_LAYER);
			vram_sw = PIP_LAYER;
			vram_addr = PIP_VRAM_ADR;
		}else{
			select_display_layer(PIP_LAYER);
			vram_sw = MAIN_LAYER;
			vram_addr = MAIN_VRAM_ADR;
		}

		display_complete = 1;
	}
}


/*------------------------------------------------------------------------
	back ground idle task
------------------------------------------------------------------------*/

void idle_task( void )
{
	for(;;){
		// do nothing
	}
}


/*========================================================================
	PhotoFrame (read from /PHOTO/IMGxxxxx.JPG)
========================================================================*/
// Photo Frame Slide showing time
static	DWORD  show_time;

BYTE PhotoFrame(void)
{
	DIR		dir;
	FATFS	fs;
	FILINFO Finfo;
	FIL	    fin;
	UINT    cnt;
	KEYCODE key;


	// mount FatFs
	f_mount(0, &fs);

	// change dir to PHOTO
	// open current dir
	if(f_chdir("/PHOTO") != FR_OK || f_opendir(&dir, "") != FR_OK){
		//putstr("directory change error.\r\n");
		goto error_exit;
	}

	// write to PIP
	select_display_layer(MAIN_LAYER);
	vram_addr = PIP_VRAM_ADR;
	vram_sw = PIP_LAYER;

	// dir read, search xxxxxxxx.JPG file
	while(f_readdir(&dir, &Finfo) == FR_OK && Finfo.fname[0]){

		if(_FS_RPATH && Finfo.fname[0] == '.') continue;
		if(Finfo.fattrib & (AM_VOL|AM_DIR)) continue;
		if(/* Finfo.fname[0] != 'I' || Finfo.fname[1] != 'M' || Finfo.fname[2] != 'G'|| */
			Finfo.fname[9] != 'J' || Finfo.fname[10] != 'P' || Finfo.fname[11] != 'G') continue;
		if(Finfo.fsize > FRMBFSZ) continue;

		// check pause 
		if(getkey() != KEY_NONE){
			if(oprPause() == 0) break;
		}

		// jpeg file open
		if(f_open(&fin, Finfo.fname, FA_READ) == FR_OK && f_read(&fin, frameBuffer, Finfo.fsize, &cnt) == FR_OK){
			// jpeg decord and display to LCD
			frame_bp = frameBuffer;
			if(jd_prepare(&jdec, image_in_func, tjpgd_pool, sizeof(tjpgd_pool), NULL) == JDR_OK){
				jd_decomp(&jdec, lcd_out_func, 0);
			}
		}

		// file close
		f_close(&fin);

		// toggle display layer
		if(vram_sw == MAIN_LAYER){
			select_display_layer(MAIN_LAYER);
			vram_sw = PIP_LAYER;
			vram_addr = PIP_VRAM_ADR;
		}else{
			select_display_layer(PIP_LAYER);
			vram_sw = MAIN_LAYER;
			vram_addr = MAIN_VRAM_ADR;
		}

		// start delayed timer
		AplTimer = show_time;
		// check pause 
		while(AplTimer){
			if((key = getkey())!= KEY_NONE){
				if(key == KEY_SET && oprPause() == 0){
					goto error_exit; // stop Photo Frame function.
				}else if(key == KEY_UP){
					break; // skip wait and go to next file
				}
			} // if
		} // while 


	} // for

error_exit:
	f_mount(0, NULL);
	return 0;
}

/*--------------------------------------------------------------------------
	set photo one frame showing time
--------------------------------------------------------------------------*/

void	set_show_time(DWORD tm)
{
	show_time = tm;
}


// *************************************************************************
// *************************************************************************
// Section:  call back functions 
// *************************************************************************
// *************************************************************************

/*========================================================================
	jpgd call back input function
    global reference: frame_bp ... jpeg image data
========================================================================*/

UINT image_in_func (JDEC* jd, BYTE* buff, UINT nbyte)
{
    if (buff) {
        /* Read bytes from image data buffer */
		memcpy(buff, frame_bp, nbyte);
		frame_bp += nbyte;
		return nbyte; 
    } else {
        /* Skip read */
		frame_bp += nbyte;
		return nbyte;
    }
}


/*========================================================================
	jpgd call back output function
    global reference: vram_addr ... point to vram buffer in LCDC
========================================================================*/

UINT lcd_out_func (JDEC* jd, void* bitmap, JRECT* rect)
{
    BYTE *src;
	long dst;
    UINT y, bws, bwd;


    // Copy the decompressed RGB rectanglar to the LCD (assuming RGB565 cfg) 
	src = (BYTE*)bitmap;
 	dst = vram_addr + 2 * (rect->top * 320 + rect->left);

 	bws = 2 * (rect->right - rect->left + 1);
 	bwd = 2 * 320;

 	for (y = rect->top; y <= rect->bottom; y++) {
		write_byte_block(dst, src, bws);
		//next line
  	    src += bws;
		dst += bwd;
  	}

	return 1;
}


/*========================================================================
	UVC Isochronous transfers call back function
	global reference:
         frame_skip    ... skip flag
         frame_wp      ... image data write point in buffer
         frame_sp      ... frame start point
         frame_size    ... size of frame
         frame_bytes   ... bytes in buffer
         frameBuffer   ... ring buffer
         FRMBFSZ       ... size of buffer
========================================================================*/

void	receive_frame_data(BYTE *data, DWORD size)
{
	DWORD length, wlen;
	BYTE *dp;

	// skip short packet
	if(size < HEADER_LEN){
		return; 
	}

	// during skip ? 
	if(frame_skip){	
		// Is data frame end? 
		if((data[1] & 2) == 2){
			frame_skip--; // skip end.
			frame_wp = frame_sp;	// reset write point to frame start.
		}
		// do nothing.
		return;
	}

	length = size - HEADER_LEN;
	dp = data + HEADER_LEN;

	// does frameBuffer not enough space?
	if((frame_bytes + length) > FRMBFSZ){
		// discard now frame
		frame_skip = 1;
		// update bytes
		frame_bytes -= frame_size;
		// clear frame size;
		frame_size = 0;
		return;	
	}

	// ring buffer over ?
	if((frame_wp - frameBuffer) + length >= FRMBFSZ){
		wlen = FRMBFSZ - (frame_wp - frameBuffer);
		memcpy((void *)frame_wp, dp, wlen);
		frame_wp = frameBuffer;  	
		frame_bytes += wlen;
		frame_size += wlen;
		dp += wlen;
		length -= wlen;
	}

	// left copy	
	if(length){
		memcpy((void *)frame_wp, dp, length);
		frame_wp += length;
		frame_bytes += length;
		frame_size += length;
	}

	// When copyed frame end update queue and next frame start
	if((data[1] & 2) == 2){
		if(frameQueue[queuePushIndex].frame_valid == 0){
			frameQueue[queuePushIndex].frame_valid = 1;
			frameQueue[queuePushIndex].frame_sp = frame_sp;
			frameQueue[queuePushIndex].frame_size = frame_size;
			queuePushIndex++;
			queuePushIndex &= QUEUE_MASK;
			IFS0SET = _IFS0_T4IF_MASK; // change to main task. 	
		}
		// set next frame start
		frame_sp = frame_wp;
		frame_size = 0;
	}	
}

/*====================================================================
    Running task was change to main task by some interrupt.
====================================================================*/

/*--------------------------------------------------------------------
	Timer 4 interrupt
--------------------------------------------------------------------*/

extern void __attribute__ (( interrupt(ipl2), vector(_TIMER_4_VECTOR) )) _T4InterruptWrapper(void);
void _T4Interrupt(void)
{
	IFS0CLR = _IFS0_T4IF_MASK; // clear interrupt flag
	// do nothing
}


/*** end of image.c ************************************************************/
